#include "../uxn.h"
#include "audio.h"
#include <stdbool.h>

/*
Copyright (c) 2021-2023 Devine Lu Linvega, Andrew Alderwick

Permission to use, copy, modify, and distribute this software for any
purpose with or without fee is hereby granted, provided that the above
copyright notice and this permission notice appear in all copies.

THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
WITH REGARD TO THIS SOFTWARE.
*/

#define SOUND_TIMER (AUDIO_BUFSIZE / SAMPLE_FREQUENCY * 1000.0f)
#define N_CHANNELS 4
#define XFADE_SAMPLES 100
#define INTERPOL_METHOD 1

typedef enum EnvStage {
    ENV_ATTACK  = (1 << 0),
    ENV_DECAY   = (1 << 1),
    ENV_SUSTAIN = (1 << 2),
    ENV_RELEASE = (1 << 3),
} EnvStage;

typedef struct Envelope {
    float a;
    float d;
    float s;
    float r;
    float vol;
    EnvStage stage;
} Envelope;

typedef struct Sample {
    Uint8 *data;
    float len;
    float pos;
    float inc;
    float loop;
    Uint8 pitch;
    Envelope env;
} Sample;

typedef struct AudioChannel {
    Sample sample;
    Sample next_sample;
    bool xfade;
    float duration;
    float vol_l;
    float vol_r;
} AudioChannel;

AudioChannel channel[N_CHANNELS];

const float tuning[109][100] = {
    {
        0.09921257f, 0.09926989f, 0.09932725f, 0.09938464f, 0.09944206f,
        0.09949952f, 0.09955701f, 0.09961453f, 0.09967209f, 0.09972968f,
        0.09978730f, 0.09984495f, 0.09990264f, 0.09996037f, 0.10001812f,
        0.10007591f, 0.10013373f, 0.10019159f, 0.10024948f, 0.10030740f,
        0.10036536f, 0.10042335f, 0.10048137f, 0.10053943f, 0.10059752f,
        0.10065565f, 0.10071380f, 0.10077199f, 0.10083022f, 0.10088848f,
        0.10094677f, 0.10100510f, 0.10106346f, 0.10112185f, 0.10118028f,
        0.10123874f, 0.10129723f, 0.10135576f, 0.10141432f, 0.10147292f,
        0.10153155f, 0.10159021f, 0.10164891f, 0.10170764f, 0.10176641f,
        0.10182521f, 0.10188404f, 0.10194291f, 0.10200181f, 0.10206075f,
        0.10211972f, 0.10217872f, 0.10223776f, 0.10229683f, 0.10235594f,
        0.10241508f, 0.10247425f, 0.10253346f, 0.10259270f, 0.10265198f,
        0.10271129f, 0.10277063f, 0.10283001f, 0.10288943f, 0.10294888f,
        0.10300836f, 0.10306788f, 0.10312743f, 0.10318701f, 0.10324663f,
        0.10330629f, 0.10336598f, 0.10342570f, 0.10348546f, 0.10354525f,
        0.10360508f, 0.10366494f, 0.10372484f, 0.10378477f, 0.10384474f,
        0.10390474f, 0.10396477f, 0.10402484f, 0.10408495f, 0.10414509f,
        0.10420526f, 0.10426547f, 0.10432571f, 0.10438599f, 0.10444630f,
        0.10450665f, 0.10456703f, 0.10462745f, 0.10468790f, 0.10474839f,
        0.10480891f, 0.10486947f, 0.10493006f, 0.10499069f, 0.10505135f,
    },
    {
        0.10511205f, 0.10517278f, 0.10523355f, 0.10529436f, 0.10535519f,
        0.10541607f, 0.10547697f, 0.10553792f, 0.10559890f, 0.10565991f,
        0.10572096f, 0.10578204f, 0.10584316f, 0.10590432f, 0.10596551f,
        0.10602674f, 0.10608800f, 0.10614929f, 0.10621062f, 0.10627199f,
        0.10633340f, 0.10639483f, 0.10645631f, 0.10651782f, 0.10657936f,
        0.10664094f, 0.10670256f, 0.10676421f, 0.10682590f, 0.10688762f,
        0.10694938f, 0.10701117f, 0.10707300f, 0.10713487f, 0.10719677f,
        0.10725871f, 0.10732068f, 0.10738269f, 0.10744473f, 0.10750681f,
        0.10756893f, 0.10763108f, 0.10769327f, 0.10775549f, 0.10781775f,
        0.10788005f, 0.10794238f, 0.10800475f, 0.10806715f, 0.10812959f,
        0.10819207f, 0.10825458f, 0.10831713f, 0.10837972f, 0.10844234f,
        0.10850499f, 0.10856769f, 0.10863042f, 0.10869318f, 0.10875598f,
        0.10881882f, 0.10888169f, 0.10894461f, 0.10900755f, 0.10907054f,
        0.10913356f, 0.10919661f, 0.10925970f, 0.10932283f, 0.10938600f,
        0.10944920f, 0.10951244f, 0.10957572f, 0.10963903f, 0.10970238f,
        0.10976576f, 0.10982918f, 0.10989264f, 0.10995613f, 0.11001967f,
        0.11008323f, 0.11014684f, 0.11021048f, 0.11027416f, 0.11033787f,
        0.11040163f, 0.11046542f, 0.11052924f, 0.11059310f, 0.11065700f,
        0.11072094f, 0.11078491f, 0.11084892f, 0.11091297f, 0.11097706f,
        0.11104118f, 0.11110534f, 0.11116953f, 0.11123376f, 0.11129803f,
    },
    {
        0.11136234f, 0.11142668f, 0.11149106f, 0.11155548f, 0.11161994f,
        0.11168443f, 0.11174896f, 0.11181353f, 0.11187813f, 0.11194278f,
        0.11200746f, 0.11207217f, 0.11213693f, 0.11220172f, 0.11226655f,
        0.11233141f, 0.11239632f, 0.11246126f, 0.11252624f, 0.11259125f,
        0.11265631f, 0.11272140f, 0.11278653f, 0.11285170f, 0.11291690f,
        0.11298214f, 0.11304742f, 0.11311274f, 0.11317810f, 0.11324349f,
        0.11330892f, 0.11337439f, 0.11343989f, 0.11350544f, 0.11357102f,
        0.11363664f, 0.11370230f, 0.11376800f, 0.11383373f, 0.11389950f,
        0.11396531f, 0.11403116f, 0.11409705f, 0.11416297f, 0.11422893f,
        0.11429493f, 0.11436097f, 0.11442705f, 0.11449316f, 0.11455931f,
        0.11462551f, 0.11469173f, 0.11475800f, 0.11482431f, 0.11489065f,
        0.11495704f, 0.11502346f, 0.11508992f, 0.11515641f, 0.11522295f,
        0.11528952f, 0.11535614f, 0.11542279f, 0.11548948f, 0.11555621f,
        0.11562297f, 0.11568978f, 0.11575662f, 0.11582351f, 0.11589043f,
        0.11595739f, 0.11602439f, 0.11609143f, 0.11615850f, 0.11622562f,
        0.11629277f, 0.11635996f, 0.11642720f, 0.11649447f, 0.11656178f,
        0.11662912f, 0.11669651f, 0.11676394f, 0.11683140f, 0.11689891f,
        0.11696645f, 0.11703403f, 0.11710165f, 0.11716931f, 0.11723701f,
        0.11730475f, 0.11737253f, 0.11744034f, 0.11750820f, 0.11757609f,
        0.11764403f, 0.11771200f, 0.11778001f, 0.11784807f, 0.11791616f,
    },
    {
        0.11798429f, 0.11805246f, 0.11812067f, 0.11818892f, 0.11825721f,
        0.11832553f, 0.11839390f, 0.11846231f, 0.11853075f, 0.11859924f,
        0.11866777f, 0.11873633f, 0.11880493f, 0.11887358f, 0.11894226f,
        0.11901099f, 0.11907975f, 0.11914855f, 0.11921740f, 0.11928628f,
        0.11935520f, 0.11942416f, 0.11949316f, 0.11956221f, 0.11963129f,
        0.11970041f, 0.11976957f, 0.11983877f, 0.11990801f, 0.11997730f,
        0.12004662f, 0.12011598f, 0.12018538f, 0.12025482f, 0.12032431f,
        0.12039383f, 0.12046339f, 0.12053299f, 0.12060263f, 0.12067232f,
        0.12074204f, 0.12081180f, 0.12088161f, 0.12095145f, 0.12102134f,
        0.12109126f, 0.12116123f, 0.12123123f, 0.12130128f, 0.12137137f,
        0.12144149f, 0.12151166f, 0.12158187f, 0.12165212f, 0.12172241f,
        0.12179274f, 0.12186311f, 0.12193352f, 0.12200397f, 0.12207446f,
        0.12214500f, 0.12221557f, 0.12228619f, 0.12235684f, 0.12242754f,
        0.12249827f, 0.12256905f, 0.12263987f, 0.12271073f, 0.12278163f,
        0.12285257f, 0.12292356f, 0.12299458f, 0.12306565f, 0.12313675f,
        0.12320790f, 0.12327909f, 0.12335032f, 0.12342159f, 0.12349290f,
        0.12356425f, 0.12363565f, 0.12370708f, 0.12377856f, 0.12385008f,
        0.12392164f, 0.12399324f, 0.12406488f, 0.12413656f, 0.12420829f,
        0.12428005f, 0.12435186f, 0.12442371f, 0.12449560f, 0.12456753f,
        0.12463951f, 0.12471152f, 0.12478358f, 0.12485568f, 0.12492782f,
    },
    {
        0.12500000f, 0.12507222f, 0.12514449f, 0.12521680f, 0.12528915f,
        0.12536154f, 0.12543397f, 0.12550644f, 0.12557896f, 0.12565152f,
        0.12572412f, 0.12579676f, 0.12586944f, 0.12594217f, 0.12601494f,
        0.12608775f, 0.12616060f, 0.12623349f, 0.12630643f, 0.12637941f,
        0.12645243f, 0.12652549f, 0.12659860f, 0.12667175f, 0.12674493f,
        0.12681817f, 0.12689144f, 0.12696476f, 0.12703812f, 0.12711152f,
        0.12718496f, 0.12725845f, 0.12733198f, 0.12740555f, 0.12747916f,
        0.12755282f, 0.12762652f, 0.12770026f, 0.12777404f, 0.12784787f,
        0.12792174f, 0.12799565f, 0.12806960f, 0.12814360f, 0.12821764f,
        0.12829172f, 0.12836585f, 0.12844002f, 0.12851423f, 0.12858848f,
        0.12866278f, 0.12873712f, 0.12881150f, 0.12888593f, 0.12896040f,
        0.12903491f, 0.12910946f, 0.12918406f, 0.12925870f, 0.12933339f,
        0.12940812f, 0.12948289f, 0.12955770f, 0.12963256f, 0.12970746f,
        0.12978240f, 0.12985739f, 0.12993242f, 0.13000749f, 0.13008261f,
        0.13015777f, 0.13023297f, 0.13030822f, 0.13038351f, 0.13045885f,
        0.13053422f, 0.13060964f, 0.13068511f, 0.13076062f, 0.13083617f,
        0.13091177f, 0.13098740f, 0.13106309f, 0.13113881f, 0.13121459f,
        0.13129040f, 0.13136626f, 0.13144216f, 0.13151811f, 0.13159410f,
        0.13167013f, 0.13174621f, 0.13182233f, 0.13189849f, 0.13197470f,
        0.13205096f, 0.13212726f, 0.13220360f, 0.13227998f, 0.13235641f,
    },
    {
        0.13243289f, 0.13250941f, 0.13258597f, 0.13266257f, 0.13273923f,
        0.13281592f, 0.13289266f, 0.13296944f, 0.13304627f, 0.13312315f,
        0.13320006f, 0.13327702f, 0.13335403f, 0.13343108f, 0.13350818f,
        0.13358532f, 0.13366250f, 0.13373973f, 0.13381700f, 0.13389432f,
        0.13397168f, 0.13404909f, 0.13412654f, 0.13420404f, 0.13428158f,
        0.13435917f, 0.13443680f, 0.13451447f, 0.13459220f, 0.13466996f,
        0.13474777f, 0.13482563f, 0.13490353f, 0.13498148f, 0.13505947f,
        0.13513750f, 0.13521558f, 0.13529371f, 0.13537188f, 0.13545010f,
        0.13552836f, 0.13560667f, 0.13568502f, 0.13576342f, 0.13584186f,
        0.13592035f, 0.13599888f, 0.13607746f, 0.13615608f, 0.13623475f,
        0.13631347f, 0.13639223f, 0.13647103f, 0.13654988f, 0.13662878f,
        0.13670772f, 0.13678671f, 0.13686575f, 0.13694483f, 0.13702395f,
        0.13710312f, 0.13718234f, 0.13726160f, 0.13734091f, 0.13742026f,
        0.13749966f, 0.13757911f, 0.13765860f, 0.13773814f, 0.13781772f,
        0.13789735f, 0.13797703f, 0.13805675f, 0.13813652f, 0.13821633f,
        0.13829619f, 0.13837610f, 0.13845605f, 0.13853605f, 0.13861609f,
        0.13869618f, 0.13877632f, 0.13885650f, 0.13893673f, 0.13901701f,
        0.13909733f, 0.13917770f, 0.13925812f, 0.13933858f, 0.13941909f,
        0.13949964f, 0.13958024f, 0.13966089f, 0.13974159f, 0.13982233f,
        0.13990312f, 0.13998395f, 0.14006483f, 0.14014576f, 0.14022673f,
    },
    {
        0.14030776f, 0.14038882f, 0.14046994f, 0.14055110f, 0.14063231f,
        0.14071357f, 0.14079487f, 0.14087622f, 0.14095762f, 0.14103906f,
        0.14112055f, 0.14120209f, 0.14128367f, 0.14136531f, 0.14144699f,
        0.14152871f, 0.14161049f, 0.14169231f, 0.14177418f, 0.14185609f,
        0.14193805f, 0.14202006f, 0.14210212f, 0.14218423f, 0.14226638f,
        0.14234858f, 0.14243083f, 0.14251312f, 0.14259546f, 0.14267785f,
        0.14276029f, 0.14284278f, 0.14292531f, 0.14300789f, 0.14309052f,
        0.14317320f, 0.14325592f, 0.14333869f, 0.14342151f, 0.14350438f,
        0.14358729f, 0.14367026f, 0.14375327f, 0.14383633f, 0.14391944f,
        0.14400259f, 0.14408579f, 0.14416904f, 0.14425234f, 0.14433569f,
        0.14441909f, 0.14450253f, 0.14458602f, 0.14466956f, 0.14475315f,
        0.14483679f, 0.14492047f, 0.14500421f, 0.14508799f, 0.14517182f,
        0.14525570f, 0.14533963f, 0.14542360f, 0.14550763f, 0.14559170f,
        0.14567582f, 0.14575999f, 0.14584421f, 0.14592848f, 0.14601279f,
        0.14609716f, 0.14618157f, 0.14626603f, 0.14635054f, 0.14643510f,
        0.14651971f, 0.14660437f, 0.14668908f, 0.14677383f, 0.14685863f,
        0.14694349f, 0.14702839f, 0.14711334f, 0.14719834f, 0.14728339f,
        0.14736849f, 0.14745364f, 0.14753884f, 0.14762408f, 0.14770938f,
        0.14779472f, 0.14788012f, 0.14796556f, 0.14805105f, 0.14813660f,
        0.14822219f, 0.14830783f, 0.14839352f, 0.14847926f, 0.14856505f,
    },
    {
        0.14865089f, 0.14873678f, 0.14882272f, 0.14890871f, 0.14899474f,
        0.14908083f, 0.14916697f, 0.14925315f, 0.14933939f, 0.14942568f,
        0.14951202f, 0.14959840f, 0.14968484f, 0.14977132f, 0.14985786f,
        0.14994445f, 0.15003108f, 0.15011777f, 0.15020451f, 0.15029129f,
        0.15037813f, 0.15046502f, 0.15055195f, 0.15063894f, 0.15072598f,
        0.15081307f, 0.15090020f, 0.15098739f, 0.15107463f, 0.15116192f,
        0.15124926f, 0.15133665f, 0.15142409f, 0.15151158f, 0.15159913f,
        0.15168672f, 0.15177436f, 0.15186205f, 0.15194980f, 0.15203759f,
        0.15212544f, 0.15221334f, 0.15230128f, 0.15238928f, 0.15247733f,
        0.15256543f, 0.15265358f, 0.15274178f, 0.15283003f, 0.15291834f,
        0.15300669f, 0.15309510f, 0.15318356f, 0.15327206f, 0.15336062f,
        0.15344923f, 0.15353789f, 0.15362661f, 0.15371537f, 0.15380419f,
        0.15389305f, 0.15398197f, 0.15407094f, 0.15415996f, 0.15424903f,
        0.15433815f, 0.15442733f, 0.15451656f, 0.15460583f, 0.15469516f,
        0.15478455f, 0.15487398f, 0.15496346f, 0.15505300f, 0.15514259f,
        0.15523223f, 0.15532192f, 0.15541166f, 0.15550146f, 0.15559130f,
        0.15568120f, 0.15577115f, 0.15586116f, 0.15595121f, 0.15604132f,
        0.15613148f, 0.15622169f, 0.15631195f, 0.15640227f, 0.15649264f,
        0.15658305f, 0.15667353f, 0.15676405f, 0.15685463f, 0.15694526f,
        0.15703594f, 0.15712667f, 0.15721746f, 0.15730830f, 0.15739919f,
    },
    {
        0.15749013f, 0.15758113f, 0.15767218f, 0.15776328f, 0.15785443f,
        0.15794564f, 0.15803690f, 0.15812821f, 0.15821957f, 0.15831099f,
        0.15840246f, 0.15849399f, 0.15858556f, 0.15867719f, 0.15876887f,
        0.15886061f, 0.15895240f, 0.15904424f, 0.15913613f, 0.15922808f,
        0.15932008f, 0.15941213f, 0.15950424f, 0.15959640f, 0.15968861f,
        0.15978088f, 0.15987320f, 0.15996557f, 0.16005800f, 0.16015048f,
        0.16024301f, 0.16033560f, 0.16042824f, 0.16052093f, 0.16061368f,
        0.16070648f, 0.16079933f, 0.16089224f, 0.16098520f, 0.16107822f,
        0.16117129f, 0.16126441f, 0.16135759f, 0.16145082f, 0.16154410f,
        0.16163744f, 0.16173083f, 0.16182428f, 0.16191778f, 0.16201134f,
        0.16210494f, 0.16219861f, 0.16229232f, 0.16238609f, 0.16247992f,
        0.16257380f, 0.16266773f, 0.16276172f, 0.16285576f, 0.16294986f,
        0.16304401f, 0.16313821f, 0.16323247f, 0.16332679f, 0.16342116f,
        0.16351558f, 0.16361006f, 0.16370459f, 0.16379918f, 0.16389382f,
        0.16398851f, 0.16408326f, 0.16417807f, 0.16427293f, 0.16436785f,
        0.16446282f, 0.16455784f, 0.16465292f, 0.16474805f, 0.16484324f,
        0.16493849f, 0.16503379f, 0.16512914f, 0.16522455f, 0.16532002f,
        0.16541554f, 0.16551111f, 0.16560674f, 0.16570243f, 0.16579817f,
        0.16589397f, 0.16598982f, 0.16608573f, 0.16618169f, 0.16627771f,
        0.16637378f, 0.16646991f, 0.16656609f, 0.16666233f, 0.16675863f,
    },
    {
        0.16685498f, 0.16695139f, 0.16704785f, 0.16714437f, 0.16724094f,
        0.16733757f, 0.16743426f, 0.16753100f, 0.16762780f, 0.16772465f,
        0.16782156f, 0.16791853f, 0.16801555f, 0.16811263f, 0.16820976f,
        0.16830695f, 0.16840420f, 0.16850150f, 0.16859886f, 0.16869627f,
        0.16879374f, 0.16889127f, 0.16898885f, 0.16908649f, 0.16918419f,
        0.16928194f, 0.16937975f, 0.16947762f, 0.16957554f, 0.16967352f,
        0.16977156f, 0.16986965f, 0.16996780f, 0.17006600f, 0.17016426f,
        0.17026258f, 0.17036096f, 0.17045939f, 0.17055788f, 0.17065643f,
        0.17075503f, 0.17085369f, 0.17095241f, 0.17105118f, 0.17115002f,
        0.17124890f, 0.17134785f, 0.17144685f, 0.17154591f, 0.17164503f,
        0.17174421f, 0.17184344f, 0.17194273f, 0.17204207f, 0.17214148f,
        0.17224094f, 0.17234046f, 0.17244004f, 0.17253967f, 0.17263936f,
        0.17273911f, 0.17283892f, 0.17293878f, 0.17303870f, 0.17313868f,
        0.17323872f, 0.17333882f, 0.17343897f, 0.17353918f, 0.17363945f,
        0.17373978f, 0.17384016f, 0.17394061f, 0.17404111f, 0.17414167f,
        0.17424228f, 0.17434296f, 0.17444369f, 0.17454448f, 0.17464533f,
        0.17474624f, 0.17484721f, 0.17494823f, 0.17504932f, 0.17515046f,
        0.17525166f, 0.17535292f, 0.17545423f, 0.17555561f, 0.17565704f,
        0.17575854f, 0.17586009f, 0.17596170f, 0.17606337f, 0.17616509f,
        0.17626688f, 0.17636873f, 0.17647063f, 0.17657259f, 0.17667461f,
    },
    {
        0.17677670f, 0.17687884f, 0.17698103f, 0.17708329f, 0.17718561f,
        0.17728798f, 0.17739042f, 0.17749291f, 0.17759547f, 0.17769808f,
        0.17780075f, 0.17790348f, 0.17800627f, 0.17810912f, 0.17821203f,
        0.17831500f, 0.17841803f, 0.17852112f, 0.17862427f, 0.17872747f,
        0.17883074f, 0.17893407f, 0.17903745f, 0.17914090f, 0.17924441f,
        0.17934797f, 0.17945160f, 0.17955528f, 0.17965903f, 0.17976283f,
        0.17986670f, 0.17997062f, 0.18007461f, 0.18017865f, 0.18028276f,
        0.18038692f, 0.18049115f, 0.18059544f, 0.18069978f, 0.18080419f,
        0.18090865f, 0.18101318f, 0.18111777f, 0.18122242f, 0.18132713f,
        0.18143189f, 0.18153672f, 0.18164161f, 0.18174656f, 0.18185158f,
        0.18195665f, 0.18206178f, 0.18216697f, 0.18227223f, 0.18237754f,
        0.18248292f, 0.18258836f, 0.18269385f, 0.18279941f, 0.18290503f,
        0.18301071f, 0.18311645f, 0.18322226f, 0.18332812f, 0.18343405f,
        0.18354003f, 0.18364608f, 0.18375219f, 0.18385836f, 0.18396459f,
        0.18407088f, 0.18417724f, 0.18428365f, 0.18439013f, 0.18449667f,
        0.18460327f, 0.18470993f, 0.18481665f, 0.18492344f, 0.18503029f,
        0.18513719f, 0.18524416f, 0.18535120f, 0.18545829f, 0.18556545f,
        0.18567266f, 0.18577994f, 0.18588729f, 0.18599469f, 0.18610216f,
        0.18620968f, 0.18631727f, 0.18642493f, 0.18653264f, 0.18664042f,
        0.18674825f, 0.18685616f, 0.18696412f, 0.18707215f, 0.18718023f,
    },
    {
        0.18728838f, 0.18739660f, 0.18750487f, 0.18761321f, 0.18772161f,
        0.18783008f, 0.18793860f, 0.18804719f, 0.18815584f, 0.18826456f,
        0.18837334f, 0.18848218f, 0.18859108f, 0.18870004f, 0.18880907f,
        0.18891817f, 0.18902732f, 0.18913654f, 0.18924582f, 0.18935516f,
        0.18946457f, 0.18957404f, 0.18968358f, 0.18979317f, 0.18990283f,
        0.19001256f, 0.19012234f, 0.19023219f, 0.19034211f, 0.19045209f,
        0.19056213f, 0.19067223f, 0.19078240f, 0.19089263f, 0.19100293f,
        0.19111329f, 0.19122371f, 0.19133420f, 0.19144475f, 0.19155536f,
        0.19166604f, 0.19177679f, 0.19188759f, 0.19199846f, 0.19210940f,
        0.19222040f, 0.19233146f, 0.19244259f, 0.19255378f, 0.19266503f,
        0.19277635f, 0.19288774f, 0.19299919f, 0.19311070f, 0.19322228f,
        0.19333392f, 0.19344562f, 0.19355740f, 0.19366923f, 0.19378113f,
        0.19389310f, 0.19400512f, 0.19411722f, 0.19422938f, 0.19434160f,
        0.19445389f, 0.19456624f, 0.19467866f, 0.19479114f, 0.19490369f,
        0.19501631f, 0.19512898f, 0.19524173f, 0.19535454f, 0.19546741f,
        0.19558035f, 0.19569335f, 0.19580642f, 0.19591956f, 0.19603276f,
        0.19614602f, 0.19625936f, 0.19637275f, 0.19648621f, 0.19659974f,
        0.19671334f, 0.19682699f, 0.19694072f, 0.19705451f, 0.19716837f,
        0.19728229f, 0.19739627f, 0.19751033f, 0.19762445f, 0.19773863f,
        0.19785288f, 0.19796720f, 0.19808158f, 0.19819603f, 0.19831055f,
    },
    {
        0.19842513f, 0.19853978f, 0.19865449f, 0.19876927f, 0.19888412f,
        0.19899903f, 0.19911401f, 0.19922906f, 0.19934417f, 0.19945935f,
        0.19957460f, 0.19968991f, 0.19980529f, 0.19992073f, 0.20003624f,
        0.20015182f, 0.20026747f, 0.20038318f, 0.20049896f, 0.20061481f,
        0.20073072f, 0.20084670f, 0.20096275f, 0.20107886f, 0.20119504f,
        0.20131129f, 0.20142761f, 0.20154399f, 0.20166044f, 0.20177696f,
        0.20189354f, 0.20201019f, 0.20212691f, 0.20224370f, 0.20236055f,
        0.20247748f, 0.20259447f, 0.20271152f, 0.20282865f, 0.20294584f,
        0.20306310f, 0.20318043f, 0.20329782f, 0.20341529f, 0.20353282f,
        0.20365042f, 0.20376808f, 0.20388582f, 0.20400362f, 0.20412149f,
        0.20423943f, 0.20435744f, 0.20447551f, 0.20459366f, 0.20471187f,
        0.20483015f, 0.20494850f, 0.20506692f, 0.20518540f, 0.20530396f,
        0.20542258f, 0.20554127f, 0.20566003f, 0.20577886f, 0.20589775f,
        0.20601672f, 0.20613575f, 0.20625486f, 0.20637403f, 0.20649327f,
        0.20661258f, 0.20673196f, 0.20685141f, 0.20697092f, 0.20709051f,
        0.20721016f, 0.20732989f, 0.20744968f, 0.20756954f, 0.20768947f,
        0.20780947f, 0.20792954f, 0.20804968f, 0.20816989f, 0.20829017f,
        0.20841052f, 0.20853094f, 0.20865142f, 0.20877198f, 0.20889261f,
        0.20901330f, 0.20913407f, 0.20925490f, 0.20937581f, 0.20949678f,
        0.20961783f, 0.20973894f, 0.20986013f, 0.20998138f, 0.21010271f,
    },
    {
        0.21022410f, 0.21034557f, 0.21046710f, 0.21058871f, 0.21071039f,
        0.21083213f, 0.21095395f, 0.21107584f, 0.21119779f, 0.21131982f,
        0.21144192f, 0.21156409f, 0.21168633f, 0.21180864f, 0.21193102f,
        0.21205347f, 0.21217599f, 0.21229859f, 0.21242125f, 0.21254398f,
        0.21266679f, 0.21278967f, 0.21291261f, 0.21303563f, 0.21315872f,
        0.21328188f, 0.21340512f, 0.21352842f, 0.21365179f, 0.21377524f,
        0.21389876f, 0.21402234f, 0.21414600f, 0.21426974f, 0.21439354f,
        0.21451741f, 0.21464136f, 0.21476538f, 0.21488947f, 0.21501363f,
        0.21513786f, 0.21526216f, 0.21538654f, 0.21551099f, 0.21563551f,
        0.21576010f, 0.21588476f, 0.21600950f, 0.21613431f, 0.21625919f,
        0.21638414f, 0.21650916f, 0.21663426f, 0.21675943f, 0.21688467f,
        0.21700999f, 0.21713537f, 0.21726083f, 0.21738636f, 0.21751196f,
        0.21763764f, 0.21776339f, 0.21788921f, 0.21801511f, 0.21814107f,
        0.21826711f, 0.21839322f, 0.21851941f, 0.21864567f, 0.21877200f,
        0.21889840f, 0.21902488f, 0.21915143f, 0.21927805f, 0.21940475f,
        0.21953152f, 0.21965836f, 0.21978528f, 0.21991227f, 0.22003933f,
        0.22016647f, 0.22029368f, 0.22042096f, 0.22054832f, 0.22067575f,
        0.22080325f, 0.22093083f, 0.22105848f, 0.22118621f, 0.22131401f,
        0.22144188f, 0.22156983f, 0.22169785f, 0.22182594f, 0.22195411f,
        0.22208235f, 0.22221067f, 0.22233906f, 0.22246753f, 0.22259607f,
    },
    {
        0.22272468f, 0.22285337f, 0.22298213f, 0.22311097f, 0.22323988f,
        0.22336886f, 0.22349792f, 0.22362706f, 0.22375627f, 0.22388555f,
        0.22401491f, 0.22414434f, 0.22427385f, 0.22440344f, 0.22453309f,
        0.22466283f, 0.22479263f, 0.22492252f, 0.22505247f, 0.22518251f,
        0.22531262f, 0.22544280f, 0.22557306f, 0.22570339f, 0.22583380f,
        0.22596428f, 0.22609484f, 0.22622548f, 0.22635619f, 0.22648698f,
        0.22661784f, 0.22674878f, 0.22687979f, 0.22701088f, 0.22714204f,
        0.22727328f, 0.22740460f, 0.22753599f, 0.22766746f, 0.22779900f,
        0.22793062f, 0.22806232f, 0.22819409f, 0.22832594f, 0.22845786f,
        0.22858986f, 0.22872194f, 0.22885409f, 0.22898632f, 0.22911863f,
        0.22925101f, 0.22938347f, 0.22951600f, 0.22964862f, 0.22978131f,
        0.22991407f, 0.23004691f, 0.23017983f, 0.23031283f, 0.23044590f,
        0.23057905f, 0.23071227f, 0.23084558f, 0.23097896f, 0.23111242f,
        0.23124595f, 0.23137956f, 0.23151325f, 0.23164702f, 0.23178086f,
        0.23191478f, 0.23204878f, 0.23218285f, 0.23231701f, 0.23245124f,
        0.23258554f, 0.23271993f, 0.23285439f, 0.23298893f, 0.23312355f,
        0.23325825f, 0.23339302f, 0.23352787f, 0.23366280f, 0.23379781f,
        0.23393290f, 0.23406806f, 0.23420330f, 0.23433862f, 0.23447402f,
        0.23460950f, 0.23474505f, 0.23488069f, 0.23501640f, 0.23515219f,
        0.23528806f, 0.23542400f, 0.23556003f, 0.23569613f, 0.23583232f,
    },
    {
        0.23596858f, 0.23610492f, 0.23624134f, 0.23637784f, 0.23651441f,
        0.23665107f, 0.23678780f, 0.23692462f, 0.23706151f, 0.23719848f,
        0.23733553f, 0.23747266f, 0.23760987f, 0.23774716f, 0.23788453f,
        0.23802197f, 0.23815950f, 0.23829711f, 0.23843479f, 0.23857256f,
        0.23871040f, 0.23884833f, 0.23898633f, 0.23912441f, 0.23926258f,
        0.23940082f, 0.23953914f, 0.23967755f, 0.23981603f, 0.23995459f,
        0.24009324f, 0.24023196f, 0.24037076f, 0.24050965f, 0.24064861f,
        0.24078766f, 0.24092678f, 0.24106598f, 0.24120527f, 0.24134464f,
        0.24148408f, 0.24162361f, 0.24176322f, 0.24190291f, 0.24204267f,
        0.24218252f, 0.24232245f, 0.24246247f, 0.24260256f, 0.24274273f,
        0.24288299f, 0.24302332f, 0.24316374f, 0.24330423f, 0.24344481f,
        0.24358547f, 0.24372621f, 0.24386704f, 0.24400794f, 0.24414893f,
        0.24428999f, 0.24443114f, 0.24457237f, 0.24471368f, 0.24485507f,
        0.24499655f, 0.24513811f, 0.24527974f, 0.24542146f, 0.24556327f,
        0.24570515f, 0.24584712f, 0.24598916f, 0.24613129f, 0.24627351f,
        0.24641580f, 0.24655818f, 0.24670063f, 0.24684318f, 0.24698580f,
        0.24712851f, 0.24727129f, 0.24741416f, 0.24755712f, 0.24770015f,
        0.24784327f, 0.24798647f, 0.24812976f, 0.24827312f, 0.24841657f,
        0.24856011f, 0.24870372f, 0.24884742f, 0.24899120f, 0.24913507f,
        0.24927901f, 0.24942304f, 0.24956716f, 0.24971136f, 0.24985564f,
    },
    {
        0.25000000f, 0.25014445f, 0.25028898f, 0.25043359f, 0.25057829f,
        0.25072307f, 0.25086794f, 0.25101289f, 0.25115792f, 0.25130304f,
        0.25144824f, 0.25159352f, 0.25173889f, 0.25188434f, 0.25202988f,
        0.25217550f, 0.25232120f, 0.25246699f, 0.25261286f, 0.25275882f,
        0.25290486f, 0.25305099f, 0.25319720f, 0.25334349f, 0.25348987f,
        0.25363633f, 0.25378288f, 0.25392952f, 0.25407623f, 0.25422304f,
        0.25436992f, 0.25451690f, 0.25466395f, 0.25481109f, 0.25495832f,
        0.25510563f, 0.25525303f, 0.25540051f, 0.25554808f, 0.25569573f,
        0.25584347f, 0.25599130f, 0.25613921f, 0.25628720f, 0.25643528f,
        0.25658345f, 0.25673170f, 0.25688003f, 0.25702846f, 0.25717697f,
        0.25732556f, 0.25747424f, 0.25762301f, 0.25777186f, 0.25792079f,
        0.25806982f, 0.25821893f, 0.25836812f, 0.25851741f, 0.25866678f,
        0.25881623f, 0.25896577f, 0.25911540f, 0.25926511f, 0.25941491f,
        0.25956480f, 0.25971478f, 0.25986484f, 0.26001498f, 0.26016522f,
        0.26031554f, 0.26046595f, 0.26061644f, 0.26076702f, 0.26091769f,
        0.26106845f, 0.26121929f, 0.26137022f, 0.26152123f, 0.26167234f,
        0.26182353f, 0.26197481f, 0.26212618f, 0.26227763f, 0.26242917f,
        0.26258080f, 0.26273252f, 0.26288432f, 0.26303621f, 0.26318819f,
        0.26334026f, 0.26349241f, 0.26364466f, 0.26379699f, 0.26394941f,
        0.26410191f, 0.26425451f, 0.26440719f, 0.26455997f, 0.26471283f,
    },
    {
        0.26486577f, 0.26501881f, 0.26517194f, 0.26532515f, 0.26547845f,
        0.26563184f, 0.26578532f, 0.26593889f, 0.26609255f, 0.26624629f,
        0.26640013f, 0.26655405f, 0.26670806f, 0.26686216f, 0.26701635f,
        0.26717063f, 0.26732500f, 0.26747946f, 0.26763400f, 0.26778864f,
        0.26794337f, 0.26809818f, 0.26825308f, 0.26840808f, 0.26856316f,
        0.26871833f, 0.26887360f, 0.26902895f, 0.26918439f, 0.26933992f,
        0.26949555f, 0.26965126f, 0.26980706f, 0.26996295f, 0.27011893f,
        0.27027500f, 0.27043117f, 0.27058742f, 0.27074376f, 0.27090019f,
        0.27105672f, 0.27121333f, 0.27137004f, 0.27152683f, 0.27168372f,
        0.27184069f, 0.27199776f, 0.27215492f, 0.27231216f, 0.27246950f,
        0.27262693f, 0.27278445f, 0.27294207f, 0.27309977f, 0.27325756f,
        0.27341545f, 0.27357343f, 0.27373149f, 0.27388965f, 0.27404790f,
        0.27420624f, 0.27436468f, 0.27452320f, 0.27468182f, 0.27484053f,
        0.27499933f, 0.27515822f, 0.27531720f, 0.27547628f, 0.27563545f,
        0.27579471f, 0.27595406f, 0.27611350f, 0.27627304f, 0.27643266f,
        0.27659238f, 0.27675220f, 0.27691210f, 0.27707210f, 0.27723219f,
        0.27739237f, 0.27755264f, 0.27771301f, 0.27787347f, 0.27803402f,
        0.27819467f, 0.27835540f, 0.27851624f, 0.27867716f, 0.27883818f,
        0.27899929f, 0.27916049f, 0.27932178f, 0.27948317f, 0.27964466f,
        0.27980623f, 0.27996790f, 0.28012966f, 0.28029152f, 0.28045347f,
    },
    {
        0.28061551f, 0.28077765f, 0.28093988f, 0.28110220f, 0.28126462f,
        0.28142713f, 0.28158974f, 0.28175244f, 0.28191523f, 0.28207812f,
        0.28224110f, 0.28240418f, 0.28256735f, 0.28273061f, 0.28289397f,
        0.28305742f, 0.28322097f, 0.28338461f, 0.28354835f, 0.28371218f,
        0.28387611f, 0.28404013f, 0.28420424f, 0.28436845f, 0.28453276f,
        0.28469716f, 0.28486165f, 0.28502624f, 0.28519093f, 0.28535571f,
        0.28552058f, 0.28568556f, 0.28585062f, 0.28601578f, 0.28618104f,
        0.28634639f, 0.28651184f, 0.28667738f, 0.28684302f, 0.28700876f,
        0.28717459f, 0.28734052f, 0.28750654f, 0.28767266f, 0.28783887f,
        0.28800518f, 0.28817159f, 0.28833809f, 0.28850469f, 0.28867138f,
        0.28883817f, 0.28900506f, 0.28917205f, 0.28933913f, 0.28950630f,
        0.28967358f, 0.28984095f, 0.29000841f, 0.29017598f, 0.29034364f,
        0.29051140f, 0.29067925f, 0.29084720f, 0.29101525f, 0.29118340f,
        0.29135164f, 0.29151998f, 0.29168842f, 0.29185695f, 0.29202558f,
        0.29219431f, 0.29236314f, 0.29253206f, 0.29270109f, 0.29287020f,
        0.29303942f, 0.29320874f, 0.29337815f, 0.29354766f, 0.29371727f,
        0.29388698f, 0.29405678f, 0.29422668f, 0.29439669f, 0.29456678f,
        0.29473698f, 0.29490728f, 0.29507767f, 0.29524817f, 0.29541876f,
        0.29558945f, 0.29576023f, 0.29593112f, 0.29610211f, 0.29627319f,
        0.29644438f, 0.29661566f, 0.29678704f, 0.29695852f, 0.29713010f,
    },
    {
        0.29730178f, 0.29747356f, 0.29764543f, 0.29781741f, 0.29798949f,
        0.29816166f, 0.29833394f, 0.29850631f, 0.29867878f, 0.29885136f,
        0.29902403f, 0.29919680f, 0.29936968f, 0.29954265f, 0.29971572f,
        0.29988889f, 0.30006217f, 0.30023554f, 0.30040901f, 0.30058259f,
        0.30075626f, 0.30093003f, 0.30110391f, 0.30127788f, 0.30145196f,
        0.30162613f, 0.30180041f, 0.30197479f, 0.30214926f, 0.30232384f,
        0.30249852f, 0.30267330f, 0.30284818f, 0.30302317f, 0.30319825f,
        0.30337344f, 0.30354872f, 0.30372411f, 0.30389960f, 0.30407519f,
        0.30425088f, 0.30442667f, 0.30460257f, 0.30477856f, 0.30495466f,
        0.30513086f, 0.30530716f, 0.30548356f, 0.30566007f, 0.30583668f,
        0.30601339f, 0.30619020f, 0.30636711f, 0.30654413f, 0.30672124f,
        0.30689846f, 0.30707579f, 0.30725321f, 0.30743074f, 0.30760837f,
        0.30778610f, 0.30796394f, 0.30814188f, 0.30831992f, 0.30849806f,
        0.30867631f, 0.30885466f, 0.30903311f, 0.30921167f, 0.30939033f,
        0.30956909f, 0.30974796f, 0.30992692f, 0.31010600f, 0.31028517f,
        0.31046445f, 0.31064384f, 0.31082332f, 0.31100291f, 0.31118261f,
        0.31136241f, 0.31154231f, 0.31172231f, 0.31190242f, 0.31208264f,
        0.31226296f, 0.31244338f, 0.31262390f, 0.31280453f, 0.31298527f,
        0.31316611f, 0.31334705f, 0.31352810f, 0.31370926f, 0.31389051f,
        0.31407188f, 0.31425334f, 0.31443492f, 0.31461659f, 0.31479838f,
    },
    {
        0.31498026f, 0.31516225f, 0.31534435f, 0.31552655f, 0.31570886f,
        0.31589128f, 0.31607379f, 0.31625642f, 0.31643915f, 0.31662198f,
        0.31680492f, 0.31698797f, 0.31717112f, 0.31735438f, 0.31753775f,
        0.31772122f, 0.31790479f, 0.31808847f, 0.31827226f, 0.31845616f,
        0.31864016f, 0.31882426f, 0.31900848f, 0.31919280f, 0.31937722f,
        0.31956176f, 0.31974640f, 0.31993114f, 0.32011599f, 0.32030095f,
        0.32048602f, 0.32067119f, 0.32085647f, 0.32104186f, 0.32122736f,
        0.32141296f, 0.32159867f, 0.32178448f, 0.32197041f, 0.32215644f,
        0.32234258f, 0.32252882f, 0.32271518f, 0.32290164f, 0.32308821f,
        0.32327488f, 0.32346167f, 0.32364856f, 0.32383556f, 0.32402267f,
        0.32420989f, 0.32439721f, 0.32458465f, 0.32477219f, 0.32495984f,
        0.32514760f, 0.32533546f, 0.32552344f, 0.32571152f, 0.32589972f,
        0.32608802f, 0.32627643f, 0.32646495f, 0.32665357f, 0.32684231f,
        0.32703116f, 0.32722011f, 0.32740918f, 0.32759835f, 0.32778763f,
        0.32797703f, 0.32816653f, 0.32835614f, 0.32854586f, 0.32873569f,
        0.32892563f, 0.32911568f, 0.32930584f, 0.32949611f, 0.32968649f,
        0.32987698f, 0.33006758f, 0.33025829f, 0.33044911f, 0.33064004f,
        0.33083108f, 0.33102223f, 0.33121349f, 0.33140486f, 0.33159634f,
        0.33178794f, 0.33197964f, 0.33217145f, 0.33236338f, 0.33255541f,
        0.33274756f, 0.33293982f, 0.33313219f, 0.33332467f, 0.33351726f,
    },
    {
        0.33370996f, 0.33390278f, 0.33409570f, 0.33428874f, 0.33448189f,
        0.33467515f, 0.33486852f, 0.33506200f, 0.33525560f, 0.33544931f,
        0.33564313f, 0.33583706f, 0.33603110f, 0.33622526f, 0.33641952f,
        0.33661390f, 0.33680839f, 0.33700300f, 0.33719772f, 0.33739254f,
        0.33758749f, 0.33778254f, 0.33797771f, 0.33817299f, 0.33836838f,
        0.33856389f, 0.33875951f, 0.33895524f, 0.33915108f, 0.33934704f,
        0.33954311f, 0.33973930f, 0.33993559f, 0.34013200f, 0.34032853f,
        0.34052517f, 0.34072192f, 0.34091878f, 0.34111576f, 0.34131286f,
        0.34151006f, 0.34170739f, 0.34190482f, 0.34210237f, 0.34230003f,
        0.34249781f, 0.34269570f, 0.34289371f, 0.34309183f, 0.34329006f,
        0.34348841f, 0.34368688f, 0.34388545f, 0.34408415f, 0.34428296f,
        0.34448188f, 0.34468092f, 0.34488007f, 0.34507934f, 0.34527872f,
        0.34547822f, 0.34567783f, 0.34587756f, 0.34607741f, 0.34627737f,
        0.34647744f, 0.34667763f, 0.34687794f, 0.34707836f, 0.34727890f,
        0.34747955f, 0.34768033f, 0.34788121f, 0.34808221f, 0.34828333f,
        0.34848457f, 0.34868592f, 0.34888738f, 0.34908897f, 0.34929067f,
        0.34949248f, 0.34969442f, 0.34989647f, 0.35009863f, 0.35030092f,
        0.35050332f, 0.35070583f, 0.35090847f, 0.35111122f, 0.35131409f,
        0.35151707f, 0.35172018f, 0.35192340f, 0.35212673f, 0.35233019f,
        0.35253376f, 0.35273745f, 0.35294126f, 0.35314519f, 0.35334923f,
    },
    {
        0.35355339f, 0.35375767f, 0.35396207f, 0.35416658f, 0.35437122f,
        0.35457597f, 0.35478084f, 0.35498583f, 0.35519093f, 0.35539616f,
        0.35560150f, 0.35580697f, 0.35601255f, 0.35621825f, 0.35642407f,
        0.35663001f, 0.35683606f, 0.35704224f, 0.35724853f, 0.35745495f,
        0.35766148f, 0.35786814f, 0.35807491f, 0.35828180f, 0.35848881f,
        0.35869594f, 0.35890319f, 0.35911056f, 0.35931805f, 0.35952566f,
        0.35973340f, 0.35994125f, 0.36014922f, 0.36035731f, 0.36056552f,
        0.36077385f, 0.36098230f, 0.36119087f, 0.36139956f, 0.36160838f,
        0.36181731f, 0.36202636f, 0.36223554f, 0.36244483f, 0.36265425f,
        0.36286379f, 0.36307345f, 0.36328323f, 0.36349313f, 0.36370315f,
        0.36391330f, 0.36412356f, 0.36433395f, 0.36454446f, 0.36475509f,
        0.36496584f, 0.36517671f, 0.36538771f, 0.36559882f, 0.36581006f,
        0.36602142f, 0.36623291f, 0.36644451f, 0.36665624f, 0.36686809f,
        0.36708006f, 0.36729216f, 0.36750438f, 0.36771672f, 0.36792918f,
        0.36814176f, 0.36835447f, 0.36856730f, 0.36878026f, 0.36899334f,
        0.36920654f, 0.36941986f, 0.36963331f, 0.36984688f, 0.37006057f,
        0.37027439f, 0.37048833f, 0.37070239f, 0.37091658f, 0.37113089f,
        0.37134533f, 0.37155989f, 0.37177457f, 0.37198938f, 0.37220431f,
        0.37241937f, 0.37263455f, 0.37284985f, 0.37306528f, 0.37328083f,
        0.37349651f, 0.37371231f, 0.37392824f, 0.37414429f, 0.37436047f,
    },
    {
        0.37457677f, 0.37479320f, 0.37500975f, 0.37522642f, 0.37544323f,
        0.37566015f, 0.37587721f, 0.37609438f, 0.37631169f, 0.37652912f,
        0.37674667f, 0.37696435f, 0.37718216f, 0.37740009f, 0.37761815f,
        0.37783633f, 0.37805464f, 0.37827308f, 0.37849164f, 0.37871033f,
        0.37892914f, 0.37914808f, 0.37936715f, 0.37958635f, 0.37980567f,
        0.38002511f, 0.38024469f, 0.38046439f, 0.38068422f, 0.38090417f,
        0.38112426f, 0.38134447f, 0.38156480f, 0.38178527f, 0.38200586f,
        0.38222658f, 0.38244742f, 0.38266840f, 0.38288950f, 0.38311073f,
        0.38333209f, 0.38355357f, 0.38377518f, 0.38399693f, 0.38421880f,
        0.38444079f, 0.38466292f, 0.38488517f, 0.38510756f, 0.38533007f,
        0.38555271f, 0.38577547f, 0.38599837f, 0.38622140f, 0.38644455f,
        0.38666784f, 0.38689125f, 0.38711479f, 0.38733846f, 0.38756226f,
        0.38778619f, 0.38801025f, 0.38823444f, 0.38845876f, 0.38868320f,
        0.38890778f, 0.38913249f, 0.38935732f, 0.38958229f, 0.38980739f,
        0.39003261f, 0.39025797f, 0.39048346f, 0.39070907f, 0.39093482f,
        0.39116070f, 0.39138671f, 0.39161285f, 0.39183912f, 0.39206552f,
        0.39229205f, 0.39251871f, 0.39274550f, 0.39297243f, 0.39319948f,
        0.39342667f, 0.39365399f, 0.39388144f, 0.39410902f, 0.39433673f,
        0.39456457f, 0.39479255f, 0.39502066f, 0.39524889f, 0.39547727f,
        0.39570577f, 0.39593440f, 0.39616317f, 0.39639207f, 0.39662110f,
    },
    {
        0.39685026f, 0.39707956f, 0.39730899f, 0.39753855f, 0.39776824f,
        0.39799807f, 0.39822803f, 0.39845812f, 0.39868834f, 0.39891870f,
        0.39914919f, 0.39937982f, 0.39961057f, 0.39984147f, 0.40007249f,
        0.40030365f, 0.40053494f, 0.40076636f, 0.40099792f, 0.40122961f,
        0.40146144f, 0.40169340f, 0.40192550f, 0.40215772f, 0.40239009f,
        0.40262258f, 0.40285521f, 0.40308798f, 0.40332088f, 0.40355391f,
        0.40378708f, 0.40402039f, 0.40425383f, 0.40448740f, 0.40472111f,
        0.40495495f, 0.40518893f, 0.40542304f, 0.40565729f, 0.40589168f,
        0.40612620f, 0.40636085f, 0.40659564f, 0.40683057f, 0.40706563f,
        0.40730083f, 0.40753617f, 0.40777164f, 0.40800724f, 0.40824298f,
        0.40847886f, 0.40871488f, 0.40895103f, 0.40918732f, 0.40942374f,
        0.40966030f, 0.40989700f, 0.41013383f, 0.41037080f, 0.41060791f,
        0.41084516f, 0.41108254f, 0.41132006f, 0.41155771f, 0.41179551f,
        0.41203344f, 0.41227151f, 0.41250971f, 0.41274806f, 0.41298654f,
        0.41322516f, 0.41346392f, 0.41370281f, 0.41394184f, 0.41418102f,
        0.41442033f, 0.41465977f, 0.41489936f, 0.41513908f, 0.41537895f,
        0.41561895f, 0.41585909f, 0.41609937f, 0.41633979f, 0.41658034f,
        0.41682104f, 0.41706187f, 0.41730285f, 0.41754396f, 0.41778521f,
        0.41802660f, 0.41826814f, 0.41850981f, 0.41875162f, 0.41899357f,
        0.41923566f, 0.41947789f, 0.41972026f, 0.41996277f, 0.42020542f,
    },
    {
        0.42044821f, 0.42069114f, 0.42093421f, 0.42117742f, 0.42142077f,
        0.42166426f, 0.42190790f, 0.42215167f, 0.42239559f, 0.42263964f,
        0.42288384f, 0.42312818f, 0.42337266f, 0.42361728f, 0.42386204f,
        0.42410694f, 0.42435199f, 0.42459717f, 0.42484250f, 0.42508797f,
        0.42533358f, 0.42557933f, 0.42582523f, 0.42607127f, 0.42631745f,
        0.42656377f, 0.42681023f, 0.42705684f, 0.42730359f, 0.42755048f,
        0.42779751f, 0.42804469f, 0.42829201f, 0.42853947f, 0.42878708f,
        0.42903483f, 0.42928272f, 0.42953075f, 0.42977893f, 0.43002725f,
        0.43027572f, 0.43052433f, 0.43077308f, 0.43102198f, 0.43127102f,
        0.43152020f, 0.43176953f, 0.43201900f, 0.43226862f, 0.43251838f,
        0.43276828f, 0.43301833f, 0.43326852f, 0.43351886f, 0.43376934f,
        0.43401997f, 0.43427074f, 0.43452166f, 0.43477272f, 0.43502393f,
        0.43527528f, 0.43552678f, 0.43577842f, 0.43603021f, 0.43628214f,
        0.43653422f, 0.43678645f, 0.43703882f, 0.43729134f, 0.43754400f,
        0.43779681f, 0.43804976f, 0.43830286f, 0.43855611f, 0.43880950f,
        0.43906304f, 0.43931673f, 0.43957056f, 0.43982454f, 0.44007866f,
        0.44033294f, 0.44058736f, 0.44084192f, 0.44109664f, 0.44135150f,
        0.44160651f, 0.44186166f, 0.44211697f, 0.44237242f, 0.44262801f,
        0.44288376f, 0.44313965f, 0.44339569f, 0.44365188f, 0.44390822f,
        0.44416471f, 0.44442134f, 0.44467812f, 0.44493505f, 0.44519213f,
    },
    {
        0.44544936f, 0.44570674f, 0.44596426f, 0.44622193f, 0.44647976f,
        0.44673773f, 0.44699585f, 0.44725412f, 0.44751254f, 0.44777110f,
        0.44802982f, 0.44828869f, 0.44854770f, 0.44880687f, 0.44906619f,
        0.44932565f, 0.44958527f, 0.44984503f, 0.45010495f, 0.45036502f,
        0.45062523f, 0.45088560f, 0.45114611f, 0.45140678f, 0.45166760f,
        0.45192857f, 0.45218969f, 0.45245096f, 0.45271238f, 0.45297395f,
        0.45323568f, 0.45349755f, 0.45375958f, 0.45402176f, 0.45428408f,
        0.45454656f, 0.45480920f, 0.45507198f, 0.45533492f, 0.45559800f,
        0.45586124f, 0.45612464f, 0.45638818f, 0.45665188f, 0.45691573f,
        0.45717973f, 0.45744388f, 0.45770819f, 0.45797265f, 0.45823726f,
        0.45850202f, 0.45876694f, 0.45903201f, 0.45929723f, 0.45956261f,
        0.45982814f, 0.46009383f, 0.46035966f, 0.46062565f, 0.46089180f,
        0.46115810f, 0.46142455f, 0.46169116f, 0.46195792f, 0.46222483f,
        0.46249190f, 0.46275912f, 0.46302650f, 0.46329403f, 0.46356172f,
        0.46382956f, 0.46409755f, 0.46436571f, 0.46463401f, 0.46490247f,
        0.46517109f, 0.46543986f, 0.46570878f, 0.46597787f, 0.46624710f,
        0.46651650f, 0.46678604f, 0.46705575f, 0.46732561f, 0.46759562f,
        0.46786580f, 0.46813612f, 0.46840661f, 0.46867725f, 0.46894805f,
        0.46921900f, 0.46949011f, 0.46976137f, 0.47003280f, 0.47030438f,
        0.47057611f, 0.47084801f, 0.47112006f, 0.47139227f, 0.47166463f,
    },
    {
        0.47193716f, 0.47220984f, 0.47248267f, 0.47275567f, 0.47302882f,
        0.47330213f, 0.47357560f, 0.47384923f, 0.47412302f, 0.47439696f,
        0.47467106f, 0.47494532f, 0.47521974f, 0.47549432f, 0.47576905f,
        0.47604395f, 0.47631900f, 0.47659421f, 0.47686958f, 0.47714511f,
        0.47742080f, 0.47769665f, 0.47797266f, 0.47824883f, 0.47852515f,
        0.47880164f, 0.47907829f, 0.47935509f, 0.47963206f, 0.47990919f,
        0.48018647f, 0.48046392f, 0.48074153f, 0.48101929f, 0.48129722f,
        0.48157531f, 0.48185356f, 0.48213197f, 0.48241054f, 0.48268927f,
        0.48296816f, 0.48324722f, 0.48352643f, 0.48380581f, 0.48408535f,
        0.48436505f, 0.48464491f, 0.48492493f, 0.48520512f, 0.48548546f,
        0.48576597f, 0.48604664f, 0.48632747f, 0.48660847f, 0.48688963f,
        0.48717095f, 0.48745243f, 0.48773407f, 0.48801588f, 0.48829785f,
        0.48857998f, 0.48886228f, 0.48914474f, 0.48942736f, 0.48971015f,
        0.48999310f, 0.49027621f, 0.49055949f, 0.49084293f, 0.49112653f,
        0.49141030f, 0.49169423f, 0.49197833f, 0.49226259f, 0.49254701f,
        0.49283160f, 0.49311635f, 0.49340127f, 0.49368635f, 0.49397160f,
        0.49425701f, 0.49454259f, 0.49482833f, 0.49511423f, 0.49540031f,
        0.49568654f, 0.49597295f, 0.49625951f, 0.49654625f, 0.49683315f,
        0.49712021f, 0.49740744f, 0.49769484f, 0.49798240f, 0.49827013f,
        0.49855803f, 0.49884609f, 0.49913432f, 0.49942271f, 0.49971127f,
    },
    {
        0.50000000f, 0.50028889f, 0.50057796f, 0.50086719f, 0.50115658f,
        0.50144614f, 0.50173587f, 0.50202577f, 0.50231584f, 0.50260607f,
        0.50289647f, 0.50318704f, 0.50347778f, 0.50376868f, 0.50405975f,
        0.50435099f, 0.50464240f, 0.50493398f, 0.50522572f, 0.50551764f,
        0.50580972f, 0.50610197f, 0.50639439f, 0.50668698f, 0.50697974f,
        0.50727267f, 0.50756576f, 0.50785903f, 0.50815247f, 0.50844607f,
        0.50873985f, 0.50903379f, 0.50932790f, 0.50962219f, 0.50991664f,
        0.51021127f, 0.51050606f, 0.51080103f, 0.51109616f, 0.51139147f,
        0.51168695f, 0.51198259f, 0.51227841f, 0.51257440f, 0.51287056f,
        0.51316689f, 0.51346339f, 0.51376007f, 0.51405691f, 0.51435393f,
        0.51465112f, 0.51494848f, 0.51524601f, 0.51554371f, 0.51584159f,
        0.51613964f, 0.51643786f, 0.51673625f, 0.51703481f, 0.51733355f,
        0.51763246f, 0.51793154f, 0.51823080f, 0.51853023f, 0.51882983f,
        0.51912960f, 0.51942955f, 0.51972967f, 0.52002997f, 0.52033043f,
        0.52063108f, 0.52093189f, 0.52123288f, 0.52153404f, 0.52183538f,
        0.52213689f, 0.52243858f, 0.52274044f, 0.52304247f, 0.52334468f,
        0.52364706f, 0.52394962f, 0.52425235f, 0.52455526f, 0.52485834f,
        0.52516160f, 0.52546503f, 0.52576864f, 0.52607242f, 0.52637638f,
        0.52668052f, 0.52698483f, 0.52728931f, 0.52759398f, 0.52789882f,
        0.52820383f, 0.52850902f, 0.52881439f, 0.52911993f, 0.52942565f,
    },
    {
        0.52973155f, 0.53003762f, 0.53034387f, 0.53065030f, 0.53095690f,
        0.53126368f, 0.53157064f, 0.53187778f, 0.53218509f, 0.53249258f,
        0.53280025f, 0.53310810f, 0.53341612f, 0.53372432f, 0.53403270f,
        0.53434126f, 0.53465000f, 0.53495891f, 0.53526801f, 0.53557728f,
        0.53588673f, 0.53619636f, 0.53650617f, 0.53681616f, 0.53712632f,
        0.53743667f, 0.53774720f, 0.53805790f, 0.53836878f, 0.53867985f,
        0.53899109f, 0.53930251f, 0.53961412f, 0.53992590f, 0.54023787f,
        0.54055001f, 0.54086233f, 0.54117484f, 0.54148752f, 0.54180039f,
        0.54211344f, 0.54242666f, 0.54274007f, 0.54305366f, 0.54336743f,
        0.54368138f, 0.54399552f, 0.54430983f, 0.54462433f, 0.54493901f,
        0.54525387f, 0.54556891f, 0.54588413f, 0.54619954f, 0.54651513f,
        0.54683090f, 0.54714685f, 0.54746299f, 0.54777930f, 0.54809581f,
        0.54841249f, 0.54872936f, 0.54904641f, 0.54936364f, 0.54968106f,
        0.54999866f, 0.55031644f, 0.55063441f, 0.55095256f, 0.55127089f,
        0.55158941f, 0.55190811f, 0.55222700f, 0.55254607f, 0.55286533f,
        0.55318477f, 0.55350439f, 0.55382420f, 0.55414419f, 0.55446437f,
        0.55478474f, 0.55510528f, 0.55542602f, 0.55574694f, 0.55606804f,
        0.55638933f, 0.55671081f, 0.55703247f, 0.55735432f, 0.55767635f,
        0.55799857f, 0.55832098f, 0.55864357f, 0.55896635f, 0.55928931f,
        0.55961246f, 0.55993580f, 0.56025933f, 0.56058304f, 0.56090694f,
    },
    {
        0.56123102f, 0.56155530f, 0.56187976f, 0.56220441f, 0.56252924f,
        0.56285427f, 0.56317948f, 0.56350488f, 0.56383046f, 0.56415624f,
        0.56448220f, 0.56480835f, 0.56513469f, 0.56546122f, 0.56578794f,
        0.56611485f, 0.56644194f, 0.56676923f, 0.56709670f, 0.56742436f,
        0.56775221f, 0.56808026f, 0.56840849f, 0.56873691f, 0.56906552f,
        0.56939432f, 0.56972331f, 0.57005249f, 0.57038186f, 0.57071142f,
        0.57104117f, 0.57137111f, 0.57170124f, 0.57203157f, 0.57236208f,
        0.57269279f, 0.57302368f, 0.57335477f, 0.57368605f, 0.57401752f,
        0.57434918f, 0.57468103f, 0.57501308f, 0.57534531f, 0.57567774f,
        0.57601036f, 0.57634317f, 0.57667618f, 0.57700938f, 0.57734277f,
        0.57767635f, 0.57801012f, 0.57834409f, 0.57867825f, 0.57901261f,
        0.57934715f, 0.57968190f, 0.58001683f, 0.58035196f, 0.58068728f,
        0.58102279f, 0.58135850f, 0.58169441f, 0.58203050f, 0.58236679f,
        0.58270328f, 0.58303996f, 0.58337683f, 0.58371390f, 0.58405117f,
        0.58438862f, 0.58472628f, 0.58506413f, 0.58540217f, 0.58574041f,
        0.58607884f, 0.58641747f, 0.58675630f, 0.58709532f, 0.58743454f,
        0.58777395f, 0.58811356f, 0.58845337f, 0.58879337f, 0.58913357f,
        0.58947396f, 0.58981456f, 0.59015535f, 0.59049633f, 0.59083751f,
        0.59117889f, 0.59152047f, 0.59186224f, 0.59220422f, 0.59254639f,
        0.59288875f, 0.59323132f, 0.59357408f, 0.59391704f, 0.59426020f,
    },
    {
        0.59460356f, 0.59494711f, 0.59529087f, 0.59563482f, 0.59597897f,
        0.59632332f, 0.59666787f, 0.59701262f, 0.59735757f, 0.59770271f,
        0.59804806f, 0.59839361f, 0.59873935f, 0.59908530f, 0.59943144f,
        0.59977779f, 0.60012433f, 0.60047108f, 0.60081802f, 0.60116517f,
        0.60151252f, 0.60186007f, 0.60220781f, 0.60255576f, 0.60290391f,
        0.60325227f, 0.60360082f, 0.60394957f, 0.60429853f, 0.60464769f,
        0.60499704f, 0.60534661f, 0.60569637f, 0.60604633f, 0.60639650f,
        0.60674687f, 0.60709744f, 0.60744822f, 0.60779919f, 0.60815037f,
        0.60850176f, 0.60885334f, 0.60920513f, 0.60955712f, 0.60990932f,
        0.61026172f, 0.61061432f, 0.61096713f, 0.61132014f, 0.61167335f,
        0.61202677f, 0.61238039f, 0.61273422f, 0.61308825f, 0.61344249f,
        0.61379693f, 0.61415157f, 0.61450643f, 0.61486148f, 0.61521674f,
        0.61557221f, 0.61592788f, 0.61628375f, 0.61663984f, 0.61699612f,
        0.61735262f, 0.61770932f, 0.61806622f, 0.61842334f, 0.61878066f,
        0.61913818f, 0.61949591f, 0.61985385f, 0.62021200f, 0.62057035f,
        0.62092891f, 0.62128767f, 0.62164665f, 0.62200583f, 0.62236522f,
        0.62272481f, 0.62308462f, 0.62344463f, 0.62380485f, 0.62416527f,
        0.62452591f, 0.62488676f, 0.62524781f, 0.62560907f, 0.62597054f,
        0.62633222f, 0.62669411f, 0.62705620f, 0.62741851f, 0.62778103f,
        0.62814375f, 0.62850669f, 0.62886983f, 0.62923319f, 0.62959675f,
    },
    {
        0.62996052f, 0.63032451f, 0.63068870f, 0.63105311f, 0.63141773f,
        0.63178255f, 0.63214759f, 0.63251284f, 0.63287830f, 0.63324397f,
        0.63360985f, 0.63397594f, 0.63434225f, 0.63470876f, 0.63507549f,
        0.63544243f, 0.63580958f, 0.63617695f, 0.63654452f, 0.63691231f,
        0.63728031f, 0.63764853f, 0.63801695f, 0.63838559f, 0.63875445f,
        0.63912351f, 0.63949279f, 0.63986228f, 0.64023199f, 0.64060191f,
        0.64097204f, 0.64134239f, 0.64171295f, 0.64208372f, 0.64245471f,
        0.64282592f, 0.64319733f, 0.64356897f, 0.64394081f, 0.64431288f,
        0.64468515f, 0.64505765f, 0.64543035f, 0.64580328f, 0.64617642f,
        0.64654977f, 0.64692334f, 0.64729712f, 0.64767113f, 0.64804534f,
        0.64841978f, 0.64879443f, 0.64916929f, 0.64954438f, 0.64991968f,
        0.65029519f, 0.65067093f, 0.65104688f, 0.65142305f, 0.65179943f,
        0.65217603f, 0.65255286f, 0.65292989f, 0.65330715f, 0.65368462f,
        0.65406232f, 0.65444023f, 0.65481835f, 0.65519670f, 0.65557527f,
        0.65595405f, 0.65633306f, 0.65671228f, 0.65709172f, 0.65747138f,
        0.65785126f, 0.65823136f, 0.65861168f, 0.65899222f, 0.65937298f,
        0.65975396f, 0.66013515f, 0.66051657f, 0.66089821f, 0.66128007f,
        0.66166215f, 0.66204446f, 0.66242698f, 0.66280972f, 0.66319269f,
        0.66357587f, 0.66395928f, 0.66434291f, 0.66472676f, 0.66511083f,
        0.66549512f, 0.66587964f, 0.66626438f, 0.66664934f, 0.66703452f,
    },
    {
        0.66741993f, 0.66780556f, 0.66819141f, 0.66857748f, 0.66896378f,
        0.66935030f, 0.66973704f, 0.67012401f, 0.67051120f, 0.67089861f,
        0.67128625f, 0.67167411f, 0.67206220f, 0.67245051f, 0.67283905f,
        0.67322780f, 0.67361679f, 0.67400600f, 0.67439543f, 0.67478509f,
        0.67517497f, 0.67556508f, 0.67595542f, 0.67634598f, 0.67673676f,
        0.67712777f, 0.67751901f, 0.67791047f, 0.67830216f, 0.67869408f,
        0.67908622f, 0.67947859f, 0.67987119f, 0.68026401f, 0.68065706f,
        0.68105033f, 0.68144384f, 0.68183757f, 0.68223153f, 0.68262571f,
        0.68302013f, 0.68341477f, 0.68380964f, 0.68420474f, 0.68460006f,
        0.68499562f, 0.68539140f, 0.68578741f, 0.68618366f, 0.68658012f,
        0.68697682f, 0.68737375f, 0.68777091f, 0.68816830f, 0.68856591f,
        0.68896376f, 0.68936183f, 0.68976014f, 0.69015868f, 0.69055744f,
        0.69095644f, 0.69135567f, 0.69175513f, 0.69215481f, 0.69255473f,
        0.69295488f, 0.69335527f, 0.69375588f, 0.69415673f, 0.69455780f,
        0.69495911f, 0.69536065f, 0.69576242f, 0.69616443f, 0.69656666f,
        0.69696913f, 0.69737183f, 0.69777477f, 0.69817793f, 0.69858133f,
        0.69898497f, 0.69938883f, 0.69979293f, 0.70019727f, 0.70060183f,
        0.70100663f, 0.70141167f, 0.70181694f, 0.70222244f, 0.70262817f,
        0.70303415f, 0.70344035f, 0.70384679f, 0.70425347f, 0.70466038f,
        0.70506752f, 0.70547490f, 0.70588252f, 0.70629037f, 0.70669846f,
    },
    {
        0.70710678f, 0.70751534f, 0.70792414f, 0.70833317f, 0.70874243f,
        0.70915194f, 0.70956168f, 0.70997166f, 0.71038187f, 0.71079232f,
        0.71120301f, 0.71161393f, 0.71202510f, 0.71243650f, 0.71284814f,
        0.71326001f, 0.71367213f, 0.71408448f, 0.71449707f, 0.71490990f,
        0.71532297f, 0.71573627f, 0.71614982f, 0.71656360f, 0.71697762f,
        0.71739189f, 0.71780639f, 0.71822113f, 0.71863611f, 0.71905133f,
        0.71946679f, 0.71988249f, 0.72029843f, 0.72071461f, 0.72113103f,
        0.72154770f, 0.72196460f, 0.72238174f, 0.72279913f, 0.72321675f,
        0.72363462f, 0.72405273f, 0.72447108f, 0.72488967f, 0.72530850f,
        0.72572758f, 0.72614690f, 0.72656646f, 0.72698626f, 0.72740630f,
        0.72782659f, 0.72824712f, 0.72866790f, 0.72908891f, 0.72951017f,
        0.72993168f, 0.73035342f, 0.73077541f, 0.73119765f, 0.73162013f,
        0.73204285f, 0.73246581f, 0.73288903f, 0.73331248f, 0.73373618f,
        0.73416013f, 0.73458432f, 0.73500875f, 0.73543343f, 0.73585836f,
        0.73628353f, 0.73670895f, 0.73713461f, 0.73756052f, 0.73798667f,
        0.73841307f, 0.73883972f, 0.73926661f, 0.73969375f, 0.74012114f,
        0.74054878f, 0.74097666f, 0.74140479f, 0.74183316f, 0.74226179f,
        0.74269066f, 0.74311978f, 0.74354914f, 0.74397876f, 0.74440862f,
        0.74483873f, 0.74526909f, 0.74569970f, 0.74613056f, 0.74656166f,
        0.74699302f, 0.74742462f, 0.74785648f, 0.74828858f, 0.74872094f,
    },
    {
        0.74915354f, 0.74958639f, 0.75001949f, 0.75045285f, 0.75088645f,
        0.75132031f, 0.75175441f, 0.75218877f, 0.75262337f, 0.75305823f,
        0.75349334f, 0.75392870f, 0.75436431f, 0.75480018f, 0.75523629f,
        0.75567266f, 0.75610928f, 0.75654615f, 0.75698328f, 0.75742065f,
        0.75785828f, 0.75829617f, 0.75873430f, 0.75917269f, 0.75961133f,
        0.76005023f, 0.76048938f, 0.76092878f, 0.76136844f, 0.76180835f,
        0.76224851f, 0.76268893f, 0.76312960f, 0.76357053f, 0.76401172f,
        0.76445315f, 0.76489485f, 0.76533679f, 0.76577900f, 0.76622146f,
        0.76666417f, 0.76710714f, 0.76755037f, 0.76799385f, 0.76843759f,
        0.76888159f, 0.76932584f, 0.76977035f, 0.77021511f, 0.77066013f,
        0.77110541f, 0.77155095f, 0.77199674f, 0.77244279f, 0.77288910f,
        0.77333567f, 0.77378250f, 0.77422958f, 0.77467692f, 0.77512452f,
        0.77557238f, 0.77602050f, 0.77646888f, 0.77691751f, 0.77736641f,
        0.77781556f, 0.77826497f, 0.77871465f, 0.77916458f, 0.77961477f,
        0.78006523f, 0.78051594f, 0.78096691f, 0.78141815f, 0.78186964f,
        0.78232140f, 0.78277342f, 0.78322569f, 0.78367823f, 0.78413104f,
        0.78458410f, 0.78503742f, 0.78549101f, 0.78594486f, 0.78639897f,
        0.78685334f, 0.78730798f, 0.78776287f, 0.78821804f, 0.78867346f,
        0.78912915f, 0.78958510f, 0.79004131f, 0.79049779f, 0.79095453f,
        0.79141154f, 0.79186881f, 0.79232634f, 0.79278414f, 0.79324220f,
    },
    {
        0.79370053f, 0.79415912f, 0.79461797f, 0.79507710f, 0.79553648f,
        0.79599614f, 0.79645605f, 0.79691624f, 0.79737669f, 0.79783740f,
        0.79829839f, 0.79875963f, 0.79922115f, 0.79968293f, 0.80014498f,
        0.80060730f, 0.80106988f, 0.80153273f, 0.80199584f, 0.80245923f,
        0.80292288f, 0.80338680f, 0.80385099f, 0.80431545f, 0.80478017f,
        0.80524517f, 0.80571043f, 0.80617596f, 0.80664176f, 0.80710783f,
        0.80757417f, 0.80804077f, 0.80850765f, 0.80897480f, 0.80944222f,
        0.80990990f, 0.81037786f, 0.81084609f, 0.81131459f, 0.81178336f,
        0.81225240f, 0.81272171f, 0.81319129f, 0.81366114f, 0.81413127f,
        0.81460166f, 0.81507233f, 0.81554327f, 0.81601448f, 0.81648597f,
        0.81695773f, 0.81742976f, 0.81790206f, 0.81837463f, 0.81884748f,
        0.81932060f, 0.81979400f, 0.82026767f, 0.82074161f, 0.82121582f,
        0.82169031f, 0.82216508f, 0.82264012f, 0.82311543f, 0.82359102f,
        0.82406688f, 0.82454302f, 0.82501943f, 0.82549612f, 0.82597308f,
        0.82645032f, 0.82692783f, 0.82740562f, 0.82788369f, 0.82836203f,
        0.82884065f, 0.82931955f, 0.82979872f, 0.83027817f, 0.83075789f,
        0.83123790f, 0.83171818f, 0.83219873f, 0.83267957f, 0.83316068f,
        0.83364208f, 0.83412375f, 0.83460569f, 0.83508792f, 0.83557042f,
        0.83605321f, 0.83653627f, 0.83701961f, 0.83750323f, 0.83798713f,
        0.83847131f, 0.83895577f, 0.83944051f, 0.83992553f, 0.84041083f,
    },
    {
        0.84089642f, 0.84138228f, 0.84186842f, 0.84235484f, 0.84284154f,
        0.84332853f, 0.84381580f, 0.84430334f, 0.84479117f, 0.84527929f,
        0.84576768f, 0.84625635f, 0.84674531f, 0.84723455f, 0.84772408f,
        0.84821388f, 0.84870397f, 0.84919434f, 0.84968500f, 0.85017594f,
        0.85066716f, 0.85115867f, 0.85165046f, 0.85214253f, 0.85263489f,
        0.85312754f, 0.85362046f, 0.85411368f, 0.85460717f, 0.85510096f,
        0.85559503f, 0.85608938f, 0.85658402f, 0.85707894f, 0.85757416f,
        0.85806965f, 0.85856544f, 0.85906151f, 0.85955786f, 0.86005451f,
        0.86055144f, 0.86104865f, 0.86154616f, 0.86204395f, 0.86254203f,
        0.86304040f, 0.86353906f, 0.86403800f, 0.86453723f, 0.86503675f,
        0.86553656f, 0.86603666f, 0.86653705f, 0.86703772f, 0.86753869f,
        0.86803994f, 0.86854149f, 0.86904332f, 0.86954544f, 0.87004786f,
        0.87055056f, 0.87105356f, 0.87155684f, 0.87206042f, 0.87256429f,
        0.87306845f, 0.87357290f, 0.87407764f, 0.87458267f, 0.87508799f,
        0.87559361f, 0.87609952f, 0.87660572f, 0.87711221f, 0.87761900f,
        0.87812608f, 0.87863345f, 0.87914112f, 0.87964908f, 0.88015733f,
        0.88066587f, 0.88117471f, 0.88168385f, 0.88219327f, 0.88270300f,
        0.88321301f, 0.88372332f, 0.88423393f, 0.88474483f, 0.88525603f,
        0.88576752f, 0.88627931f, 0.88679139f, 0.88730377f, 0.88781644f,
        0.88832941f, 0.88884268f, 0.88935625f, 0.88987011f, 0.89038426f,
    },
    {
        0.89089872f, 0.89141347f, 0.89192852f, 0.89244387f, 0.89295951f,
        0.89347545f, 0.89399169f, 0.89450823f, 0.89502507f, 0.89554221f,
        0.89605964f, 0.89657738f, 0.89709541f, 0.89761374f, 0.89813237f,
        0.89865130f, 0.89917054f, 0.89969007f, 0.90020990f, 0.90073003f,
        0.90125046f, 0.90177120f, 0.90229223f, 0.90281356f, 0.90333520f,
        0.90385714f, 0.90437938f, 0.90490192f, 0.90542476f, 0.90594791f,
        0.90647135f, 0.90699510f, 0.90751916f, 0.90804351f, 0.90856817f,
        0.90909313f, 0.90961839f, 0.91014396f, 0.91066983f, 0.91119601f,
        0.91172249f, 0.91224927f, 0.91277636f, 0.91330375f, 0.91383145f,
        0.91435945f, 0.91488776f, 0.91541637f, 0.91594529f, 0.91647451f,
        0.91700404f, 0.91753388f, 0.91806402f, 0.91859447f, 0.91912522f,
        0.91965628f, 0.92018765f, 0.92071933f, 0.92125131f, 0.92178360f,
        0.92231619f, 0.92284910f, 0.92338231f, 0.92391583f, 0.92444966f,
        0.92498380f, 0.92551824f, 0.92605300f, 0.92658806f, 0.92712343f,
        0.92765912f, 0.92819511f, 0.92873141f, 0.92926802f, 0.92980494f,
        0.93034217f, 0.93087972f, 0.93141757f, 0.93195573f, 0.93249421f,
        0.93303299f, 0.93357209f, 0.93411150f, 0.93465122f, 0.93519125f,
        0.93573159f, 0.93627225f, 0.93681322f, 0.93735450f, 0.93789609f,
        0.93843800f, 0.93898022f, 0.93952275f, 0.94006560f, 0.94060876f,
        0.94115223f, 0.94169602f, 0.94224012f, 0.94278454f, 0.94332927f,
    },
    {
        0.94387431f, 0.94441967f, 0.94496535f, 0.94551134f, 0.94605765f,
        0.94660427f, 0.94715121f, 0.94769846f, 0.94824603f, 0.94879392f,
        0.94934212f, 0.94989064f, 0.95043948f, 0.95098863f, 0.95153810f,
        0.95208789f, 0.95263800f, 0.95318842f, 0.95373916f, 0.95429023f,
        0.95484160f, 0.95539330f, 0.95594532f, 0.95649765f, 0.95705031f,
        0.95760328f, 0.95815657f, 0.95871019f, 0.95926412f, 0.95981837f,
        0.96037295f, 0.96092784f, 0.96148305f, 0.96203859f, 0.96259444f,
        0.96315062f, 0.96370712f, 0.96426394f, 0.96482108f, 0.96537854f,
        0.96593633f, 0.96649444f, 0.96705287f, 0.96761162f, 0.96817070f,
        0.96873009f, 0.96928982f, 0.96984986f, 0.97041023f, 0.97097092f,
        0.97153194f, 0.97209328f, 0.97265495f, 0.97321694f, 0.97377925f,
        0.97434189f, 0.97490486f, 0.97546815f, 0.97603176f, 0.97659570f,
        0.97715997f, 0.97772456f, 0.97828948f, 0.97885473f, 0.97942030f,
        0.97998620f, 0.98055242f, 0.98111897f, 0.98168586f, 0.98225306f,
        0.98282060f, 0.98338846f, 0.98395665f, 0.98452517f, 0.98509402f,
        0.98566320f, 0.98623270f, 0.98680254f, 0.98737270f, 0.98794320f,
        0.98851402f, 0.98908517f, 0.98965666f, 0.99022847f, 0.99080061f,
        0.99137309f, 0.99194589f, 0.99251903f, 0.99309250f, 0.99366629f,
        0.99424042f, 0.99481489f, 0.99538968f, 0.99596480f, 0.99654026f,
        0.99711605f, 0.99769218f, 0.99826863f, 0.99884542f, 0.99942254f,
    },
    {
        1.00000000f, 1.00057779f, 1.00115591f, 1.00173437f, 1.00231316f,
        1.00289229f, 1.00347175f, 1.00405154f, 1.00463167f, 1.00521214f,
        1.00579294f, 1.00637408f, 1.00695555f, 1.00753736f, 1.00811950f,
        1.00870198f, 1.00928480f, 1.00986796f, 1.01045145f, 1.01103527f,
        1.01161944f, 1.01220394f, 1.01278878f, 1.01337396f, 1.01395948f,
        1.01454533f, 1.01513153f, 1.01571806f, 1.01630493f, 1.01689214f,
        1.01747969f, 1.01806758f, 1.01865581f, 1.01924438f, 1.01983329f,
        1.02042254f, 1.02101213f, 1.02160206f, 1.02219233f, 1.02278294f,
        1.02337389f, 1.02396519f, 1.02455682f, 1.02514880f, 1.02574112f,
        1.02633378f, 1.02692679f, 1.02752014f, 1.02811383f, 1.02870786f,
        1.02930224f, 1.02989696f, 1.03049202f, 1.03108743f, 1.03168318f,
        1.03227928f, 1.03287572f, 1.03347250f, 1.03406963f, 1.03466710f,
        1.03526492f, 1.03586309f, 1.03646160f, 1.03706046f, 1.03765966f,
        1.03825921f, 1.03885910f, 1.03945935f, 1.04005993f, 1.04066087f,
        1.04126215f, 1.04186378f, 1.04246576f, 1.04306809f, 1.04367076f,
        1.04427378f, 1.04487715f, 1.04548087f, 1.04608494f, 1.04668936f,
        1.04729412f, 1.04789924f, 1.04850470f, 1.04911052f, 1.04971668f,
        1.05032320f, 1.05093006f, 1.05153728f, 1.05214485f, 1.05275277f,
        1.05336104f, 1.05396966f, 1.05457863f, 1.05518795f, 1.05579763f,
        1.05640766f, 1.05701804f, 1.05762877f, 1.05823986f, 1.05885130f,
    },
    {
        1.05946309f, 1.06007524f, 1.06068774f, 1.06130060f, 1.06191380f,
        1.06252737f, 1.06314128f, 1.06375556f, 1.06437018f, 1.06498516f,
        1.06560050f, 1.06621619f, 1.06683224f, 1.06744865f, 1.06806541f,
        1.06868253f, 1.06930000f, 1.06991783f, 1.07053602f, 1.07115456f,
        1.07177346f, 1.07239272f, 1.07301234f, 1.07363231f, 1.07425265f,
        1.07487334f, 1.07549439f, 1.07611580f, 1.07673757f, 1.07735970f,
        1.07798218f, 1.07860503f, 1.07922824f, 1.07985180f, 1.08047573f,
        1.08110002f, 1.08172467f, 1.08234968f, 1.08297505f, 1.08360078f,
        1.08422687f, 1.08485333f, 1.08548014f, 1.08610732f, 1.08673486f,
        1.08736277f, 1.08799103f, 1.08861966f, 1.08924866f, 1.08987801f,
        1.09050773f, 1.09113782f, 1.09176826f, 1.09239908f, 1.09303025f,
        1.09366179f, 1.09429370f, 1.09492597f, 1.09555861f, 1.09619161f,
        1.09682498f, 1.09745871f, 1.09809281f, 1.09872728f, 1.09936211f,
        1.09999731f, 1.10063288f, 1.10126881f, 1.10190512f, 1.10254179f,
        1.10317882f, 1.10381623f, 1.10445400f, 1.10509214f, 1.10573065f,
        1.10636953f, 1.10700878f, 1.10764840f, 1.10828839f, 1.10892874f,
        1.10956947f, 1.11021057f, 1.11085204f, 1.11149388f, 1.11213609f,
        1.11277867f, 1.11342162f, 1.11406494f, 1.11470864f, 1.11535270f,
        1.11599714f, 1.11664195f, 1.11728714f, 1.11793269f, 1.11857862f,
        1.11922493f, 1.11987160f, 1.12051865f, 1.12116608f, 1.12181388f,
    },
    {
        1.12246205f, 1.12311060f, 1.12375952f, 1.12440881f, 1.12505848f,
        1.12570853f, 1.12635895f, 1.12700975f, 1.12766093f, 1.12831248f,
        1.12896440f, 1.12961671f, 1.13026939f, 1.13092245f, 1.13157588f,
        1.13222969f, 1.13288389f, 1.13353845f, 1.13419340f, 1.13484873f,
        1.13550443f, 1.13616051f, 1.13681697f, 1.13747381f, 1.13813103f,
        1.13878863f, 1.13944661f, 1.14010498f, 1.14076372f, 1.14142284f,
        1.14208234f, 1.14274222f, 1.14340249f, 1.14406313f, 1.14472416f,
        1.14538557f, 1.14604736f, 1.14670954f, 1.14737209f, 1.14803503f,
        1.14869835f, 1.14936206f, 1.15002615f, 1.15069062f, 1.15135548f,
        1.15202072f, 1.15268635f, 1.15335236f, 1.15401875f, 1.15468553f,
        1.15535270f, 1.15602025f, 1.15668818f, 1.15735651f, 1.15802521f,
        1.15869431f, 1.15936379f, 1.16003366f, 1.16070391f, 1.16137456f,
        1.16204559f, 1.16271700f, 1.16338881f, 1.16406100f, 1.16473359f,
        1.16540656f, 1.16607992f, 1.16675367f, 1.16742780f, 1.16810233f,
        1.16877725f, 1.16945256f, 1.17012825f, 1.17080434f, 1.17148082f,
        1.17215769f, 1.17283495f, 1.17351260f, 1.17419064f, 1.17486908f,
        1.17554791f, 1.17622713f, 1.17690674f, 1.17758674f, 1.17826714f,
        1.17894793f, 1.17962911f, 1.18031069f, 1.18099266f, 1.18167503f,
        1.18235779f, 1.18304094f, 1.18372449f, 1.18440843f, 1.18509277f,
        1.18577751f, 1.18646263f, 1.18714816f, 1.18783408f, 1.18852040f,
    },
    {
        1.18920712f, 1.18989423f, 1.19058173f, 1.19126964f, 1.19195794f,
        1.19264664f, 1.19333574f, 1.19402524f, 1.19471514f, 1.19540543f,
        1.19609612f, 1.19678721f, 1.19747870f, 1.19817060f, 1.19886289f,
        1.19955558f, 1.20024867f, 1.20094216f, 1.20163605f, 1.20233034f,
        1.20302504f, 1.20372013f, 1.20441563f, 1.20511153f, 1.20580783f,
        1.20650453f, 1.20720164f, 1.20789914f, 1.20859706f, 1.20929537f,
        1.20999409f, 1.21069321f, 1.21139274f, 1.21209267f, 1.21279300f,
        1.21349374f, 1.21419488f, 1.21489643f, 1.21559839f, 1.21630075f,
        1.21700351f, 1.21770669f, 1.21841026f, 1.21911425f, 1.21981864f,
        1.22052344f, 1.22122864f, 1.22193426f, 1.22264028f, 1.22334671f,
        1.22405354f, 1.22476079f, 1.22546844f, 1.22617651f, 1.22688498f,
        1.22759386f, 1.22830315f, 1.22901285f, 1.22972296f, 1.23043348f,
        1.23114441f, 1.23185576f, 1.23256751f, 1.23327967f, 1.23399225f,
        1.23470524f, 1.23541864f, 1.23613245f, 1.23684667f, 1.23756131f,
        1.23827636f, 1.23899182f, 1.23970770f, 1.24042399f, 1.24114069f,
        1.24185781f, 1.24257534f, 1.24329329f, 1.24401165f, 1.24473043f,
        1.24544962f, 1.24616923f, 1.24688925f, 1.24760969f, 1.24833055f,
        1.24905182f, 1.24977351f, 1.25049562f, 1.25121814f, 1.25194108f,
        1.25266444f, 1.25338821f, 1.25411241f, 1.25483702f, 1.25556205f,
        1.25628750f, 1.25701337f, 1.25773966f, 1.25846637f, 1.25919350f,
    },
    {
        1.25992105f, 1.26064902f, 1.26137741f, 1.26210622f, 1.26283545f,
        1.26356510f, 1.26429518f, 1.26502568f, 1.26575659f, 1.26648793f,
        1.26721970f, 1.26795188f, 1.26868449f, 1.26941753f, 1.27015098f,
        1.27088486f, 1.27161917f, 1.27235389f, 1.27308905f, 1.27382462f,
        1.27456063f, 1.27529706f, 1.27603391f, 1.27677119f, 1.27750889f,
        1.27824702f, 1.27898558f, 1.27972457f, 1.28046398f, 1.28120382f,
        1.28194408f, 1.28268478f, 1.28342590f, 1.28416745f, 1.28490943f,
        1.28565183f, 1.28639467f, 1.28713793f, 1.28788163f, 1.28862575f,
        1.28937031f, 1.29011529f, 1.29086071f, 1.29160655f, 1.29235283f,
        1.29309954f, 1.29384668f, 1.29459425f, 1.29534225f, 1.29609069f,
        1.29683955f, 1.29758885f, 1.29833859f, 1.29908875f, 1.29983935f,
        1.30059039f, 1.30134186f, 1.30209376f, 1.30284609f, 1.30359886f,
        1.30435207f, 1.30510571f, 1.30585979f, 1.30661430f, 1.30736925f,
        1.30812463f, 1.30888045f, 1.30963671f, 1.31039340f, 1.31115054f,
        1.31190810f, 1.31266611f, 1.31342456f, 1.31418344f, 1.31494276f,
        1.31570252f, 1.31646272f, 1.31722336f, 1.31798444f, 1.31874595f,
        1.31950791f, 1.32027031f, 1.32103315f, 1.32179643f, 1.32256015f,
        1.32332431f, 1.32408891f, 1.32485396f, 1.32561944f, 1.32638537f,
        1.32715174f, 1.32791856f, 1.32868581f, 1.32945351f, 1.33022166f,
        1.33099025f, 1.33175928f, 1.33252876f, 1.33329868f, 1.33406904f,
    },
    {
        1.33483985f, 1.33561111f, 1.33638281f, 1.33715496f, 1.33792755f,
        1.33870060f, 1.33947408f, 1.34024802f, 1.34102240f, 1.34179723f,
        1.34257250f, 1.34334823f, 1.34412440f, 1.34490102f, 1.34567809f,
        1.34645561f, 1.34723358f, 1.34801199f, 1.34879086f, 1.34957018f,
        1.35034995f, 1.35113016f, 1.35191083f, 1.35269195f, 1.35347352f,
        1.35425555f, 1.35503802f, 1.35582095f, 1.35660433f, 1.35738816f,
        1.35817244f, 1.35895718f, 1.35974237f, 1.36052802f, 1.36131412f,
        1.36210067f, 1.36288768f, 1.36367514f, 1.36446306f, 1.36525143f,
        1.36604026f, 1.36682954f, 1.36761928f, 1.36840948f, 1.36920013f,
        1.36999124f, 1.37078280f, 1.37157483f, 1.37236731f, 1.37316025f,
        1.37395365f, 1.37474750f, 1.37554182f, 1.37633659f, 1.37713182f,
        1.37792752f, 1.37872367f, 1.37952028f, 1.38031735f, 1.38111489f,
        1.38191288f, 1.38271133f, 1.38351025f, 1.38430963f, 1.38510947f,
        1.38590977f, 1.38671053f, 1.38751176f, 1.38831345f, 1.38911560f,
        1.38991822f, 1.39072130f, 1.39152484f, 1.39232885f, 1.39313333f,
        1.39393826f, 1.39474367f, 1.39554953f, 1.39635587f, 1.39716267f,
        1.39796993f, 1.39877767f, 1.39958587f, 1.40039453f, 1.40120366f,
        1.40201327f, 1.40282333f, 1.40363387f, 1.40444488f, 1.40525635f,
        1.40606829f, 1.40688070f, 1.40769358f, 1.40850693f, 1.40932076f,
        1.41013505f, 1.41094981f, 1.41176504f, 1.41258074f, 1.41339692f,
    },
    {
        1.41421356f, 1.41503068f, 1.41584827f, 1.41666633f, 1.41748487f,
        1.41830388f, 1.41912336f, 1.41994331f, 1.42076374f, 1.42158464f,
        1.42240602f, 1.42322787f, 1.42405020f, 1.42487300f, 1.42569627f,
        1.42652003f, 1.42734425f, 1.42816896f, 1.42899414f, 1.42981980f,
        1.43064593f, 1.43147254f, 1.43229963f, 1.43312720f, 1.43395525f,
        1.43478377f, 1.43561278f, 1.43644226f, 1.43727222f, 1.43810266f,
        1.43893358f, 1.43976498f, 1.44059686f, 1.44142922f, 1.44226207f,
        1.44309539f, 1.44392920f, 1.44476348f, 1.44559825f, 1.44643350f,
        1.44726924f, 1.44810545f, 1.44894215f, 1.44977934f, 1.45061701f,
        1.45145516f, 1.45229379f, 1.45313291f, 1.45397252f, 1.45481261f,
        1.45565318f, 1.45649424f, 1.45733579f, 1.45817782f, 1.45902034f,
        1.45986335f, 1.46070684f, 1.46155083f, 1.46239529f, 1.46324025f,
        1.46408570f, 1.46493163f, 1.46577805f, 1.46662496f, 1.46747236f,
        1.46832025f, 1.46916863f, 1.47001750f, 1.47086686f, 1.47171672f,
        1.47256706f, 1.47341789f, 1.47426922f, 1.47512103f, 1.47597334f,
        1.47682615f, 1.47767944f, 1.47853323f, 1.47938751f, 1.48024228f,
        1.48109755f, 1.48195331f, 1.48280957f, 1.48366632f, 1.48452357f,
        1.48538131f, 1.48623955f, 1.48709828f, 1.48795751f, 1.48881724f,
        1.48967746f, 1.49053818f, 1.49139940f, 1.49226112f, 1.49312333f,
        1.49398604f, 1.49484925f, 1.49571296f, 1.49657716f, 1.49744187f,
    },
    {
        1.49830708f, 1.49917278f, 1.50003899f, 1.50090570f, 1.50177290f,
        1.50264061f, 1.50350882f, 1.50437753f, 1.50524675f, 1.50611646f,
        1.50698668f, 1.50785740f, 1.50872863f, 1.50960035f, 1.51047259f,
        1.51134532f, 1.51221856f, 1.51309230f, 1.51396655f, 1.51484131f,
        1.51571657f, 1.51659233f, 1.51746860f, 1.51834538f, 1.51922266f,
        1.52010046f, 1.52097875f, 1.52185756f, 1.52273687f, 1.52361669f,
        1.52449702f, 1.52537786f, 1.52625921f, 1.52714107f, 1.52802343f,
        1.52890631f, 1.52978969f, 1.53067359f, 1.53155800f, 1.53244292f,
        1.53332834f, 1.53421429f, 1.53510074f, 1.53598770f, 1.53687518f,
        1.53776317f, 1.53865168f, 1.53954069f, 1.54043022f, 1.54132027f,
        1.54221083f, 1.54310190f, 1.54399349f, 1.54488559f, 1.54577821f,
        1.54667134f, 1.54756499f, 1.54845916f, 1.54935384f, 1.55024904f,
        1.55114476f, 1.55204100f, 1.55293775f, 1.55383502f, 1.55473281f,
        1.55563112f, 1.55652995f, 1.55742929f, 1.55832916f, 1.55922955f,
        1.56013045f, 1.56103188f, 1.56193383f, 1.56283630f, 1.56373929f,
        1.56464280f, 1.56554683f, 1.56645139f, 1.56735647f, 1.56826207f,
        1.56916820f, 1.57007484f, 1.57098202f, 1.57188971f, 1.57279794f,
        1.57370668f, 1.57461595f, 1.57552575f, 1.57643607f, 1.57734692f,
        1.57825829f, 1.57917020f, 1.58008262f, 1.58099558f, 1.58190906f,
        1.58282307f, 1.58373761f, 1.58465268f, 1.58556827f, 1.58648440f,
    },
    {
        1.58740105f, 1.58831824f, 1.58923595f, 1.59015419f, 1.59107297f,
        1.59199227f, 1.59291211f, 1.59383248f, 1.59475338f, 1.59567481f,
        1.59659677f, 1.59751927f, 1.59844230f, 1.59936586f, 1.60028996f,
        1.60121459f, 1.60213976f, 1.60306545f, 1.60399169f, 1.60491846f,
        1.60584576f, 1.60677360f, 1.60770198f, 1.60863089f, 1.60956034f,
        1.61049033f, 1.61142086f, 1.61235192f, 1.61328352f, 1.61421566f,
        1.61514833f, 1.61608155f, 1.61701530f, 1.61794960f, 1.61888443f,
        1.61981981f, 1.62075572f, 1.62169218f, 1.62262917f, 1.62356671f,
        1.62450479f, 1.62544341f, 1.62638258f, 1.62732229f, 1.62826254f,
        1.62920333f, 1.63014466f, 1.63108655f, 1.63202897f, 1.63297194f,
        1.63391545f, 1.63485951f, 1.63580412f, 1.63674927f, 1.63769496f,
        1.63864121f, 1.63958800f, 1.64053533f, 1.64148322f, 1.64243165f,
        1.64338063f, 1.64433016f, 1.64528023f, 1.64623086f, 1.64718203f,
        1.64813376f, 1.64908603f, 1.65003886f, 1.65099223f, 1.65194616f,
        1.65290064f, 1.65385566f, 1.65481125f, 1.65576738f, 1.65672406f,
        1.65768130f, 1.65863909f, 1.65959744f, 1.66055633f, 1.66151579f,
        1.66247579f, 1.66343635f, 1.66439747f, 1.66535914f, 1.66632137f,
        1.66728415f, 1.66824749f, 1.66921139f, 1.67017584f, 1.67114085f,
        1.67210642f, 1.67307254f, 1.67403923f, 1.67500647f, 1.67597427f,
        1.67694263f, 1.67791155f, 1.67888103f, 1.67985107f, 1.68082167f,
    },
    {
        1.68179283f, 1.68276455f, 1.68373684f, 1.68470968f, 1.68568309f,
        1.68665706f, 1.68763159f, 1.68860669f, 1.68958235f, 1.69055857f,
        1.69153536f, 1.69251271f, 1.69349062f, 1.69446911f, 1.69544815f,
        1.69642776f, 1.69740794f, 1.69838869f, 1.69937000f, 1.70035188f,
        1.70133432f, 1.70231734f, 1.70330092f, 1.70428507f, 1.70526978f,
        1.70625507f, 1.70724093f, 1.70822735f, 1.70921435f, 1.71020191f,
        1.71119005f, 1.71217876f, 1.71316804f, 1.71415789f, 1.71514831f,
        1.71613931f, 1.71713087f, 1.71812301f, 1.71911573f, 1.72010901f,
        1.72110287f, 1.72209731f, 1.72309232f, 1.72408790f, 1.72508406f,
        1.72608080f, 1.72707811f, 1.72807600f, 1.72907446f, 1.73007350f,
        1.73107312f, 1.73207332f, 1.73307409f, 1.73407544f, 1.73507737f,
        1.73607988f, 1.73708297f, 1.73808664f, 1.73909089f, 1.74009572f,
        1.74110113f, 1.74210712f, 1.74311369f, 1.74412084f, 1.74512858f,
        1.74613689f, 1.74714579f, 1.74815527f, 1.74916534f, 1.75017599f,
        1.75118722f, 1.75219904f, 1.75321144f, 1.75422443f, 1.75523800f,
        1.75625216f, 1.75726690f, 1.75828223f, 1.75929815f, 1.76031466f,
        1.76133175f, 1.76234943f, 1.76336769f, 1.76438655f, 1.76540599f,
        1.76642603f, 1.76744665f, 1.76846786f, 1.76948966f, 1.77051205f,
        1.77153504f, 1.77255861f, 1.77358278f, 1.77460754f, 1.77563289f,
        1.77665883f, 1.77768536f, 1.77871249f, 1.77974021f, 1.78076853f,
    },
    {
        1.78179744f, 1.78282694f, 1.78385704f, 1.78488773f, 1.78591902f,
        1.78695091f, 1.78798339f, 1.78901647f, 1.79005014f, 1.79108441f,
        1.79211928f, 1.79315475f, 1.79419082f, 1.79522748f, 1.79626475f,
        1.79730261f, 1.79834107f, 1.79938013f, 1.80041980f, 1.80146006f,
        1.80250093f, 1.80354239f, 1.80458446f, 1.80562713f, 1.80667040f,
        1.80771428f, 1.80875876f, 1.80980384f, 1.81084952f, 1.81189581f,
        1.81294271f, 1.81399021f, 1.81503831f, 1.81608702f, 1.81713634f,
        1.81818626f, 1.81923679f, 1.82028792f, 1.82133967f, 1.82239202f,
        1.82344498f, 1.82449854f, 1.82555272f, 1.82660751f, 1.82766290f,
        1.82871890f, 1.82977552f, 1.83083274f, 1.83189058f, 1.83294903f,
        1.83400809f, 1.83506776f, 1.83612804f, 1.83718894f, 1.83825044f,
        1.83931257f, 1.84037530f, 1.84143865f, 1.84250261f, 1.84356719f,
        1.84463239f, 1.84569820f, 1.84676462f, 1.84783166f, 1.84889932f,
        1.84996760f, 1.85103649f, 1.85210600f, 1.85317612f, 1.85424687f,
        1.85531823f, 1.85639022f, 1.85746282f, 1.85853604f, 1.85960989f,
        1.86068435f, 1.86175943f, 1.86283514f, 1.86391146f, 1.86498841f,
        1.86606598f, 1.86714418f, 1.86822299f, 1.86930243f, 1.87038250f,
        1.87146318f, 1.87254449f, 1.87362643f, 1.87470899f, 1.87579218f,
        1.87687599f, 1.87796043f, 1.87904550f, 1.88013119f, 1.88121751f,
        1.88230446f, 1.88339203f, 1.88448024f, 1.88556907f, 1.88665853f,
    },
    {
        1.88774863f, 1.88883935f, 1.88993070f, 1.89102268f, 1.89211529f,
        1.89320854f, 1.89430241f, 1.89539692f, 1.89649206f, 1.89758784f,
        1.89868424f, 1.89978128f, 1.90087896f, 1.90197726f, 1.90307621f,
        1.90417578f, 1.90527600f, 1.90637684f, 1.90747833f, 1.90858045f,
        1.90968321f, 1.91078660f, 1.91189064f, 1.91299531f, 1.91410061f,
        1.91520656f, 1.91631315f, 1.91742037f, 1.91852824f, 1.91963674f,
        1.92074589f, 1.92185568f, 1.92296610f, 1.92407717f, 1.92518889f,
        1.92630124f, 1.92741424f, 1.92852788f, 1.92964216f, 1.93075709f,
        1.93187266f, 1.93298887f, 1.93410573f, 1.93522324f, 1.93634139f,
        1.93746019f, 1.93857963f, 1.93969972f, 1.94082046f, 1.94194185f,
        1.94306388f, 1.94418656f, 1.94530989f, 1.94643387f, 1.94755850f,
        1.94868378f, 1.94980971f, 1.95093629f, 1.95206352f, 1.95319140f,
        1.95431994f, 1.95544912f, 1.95657896f, 1.95770945f, 1.95884060f,
        1.95997239f, 1.96110484f, 1.96223795f, 1.96337171f, 1.96450613f,
        1.96564120f, 1.96677692f, 1.96791331f, 1.96905035f, 1.97018804f,
        1.97132640f, 1.97246541f, 1.97360508f, 1.97474541f, 1.97588639f,
        1.97702804f, 1.97817035f, 1.97931331f, 1.98045694f, 1.98160123f,
        1.98274617f, 1.98389178f, 1.98503806f, 1.98618499f, 1.98733259f,
        1.98848085f, 1.98962977f, 1.99077936f, 1.99192961f, 1.99308053f,
        1.99423211f, 1.99538435f, 1.99653727f, 1.99769084f, 1.99884509f,
    },
    {
        2.00000000f, 2.00115558f, 2.00231183f, 2.00346874f, 2.00462632f,
        2.00578458f, 2.00694350f, 2.00810309f, 2.00926335f, 2.01042428f,
        2.01158588f, 2.01274816f, 2.01391110f, 2.01507472f, 2.01623901f,
        2.01740397f, 2.01856960f, 2.01973591f, 2.02090289f, 2.02207055f,
        2.02323888f, 2.02440789f, 2.02557757f, 2.02674793f, 2.02791896f,
        2.02909067f, 2.03026306f, 2.03143612f, 2.03260986f, 2.03378428f,
        2.03495938f, 2.03613516f, 2.03731162f, 2.03848876f, 2.03966657f,
        2.04084507f, 2.04202425f, 2.04320411f, 2.04438465f, 2.04556588f,
        2.04674778f, 2.04793037f, 2.04911365f, 2.05029760f, 2.05148224f,
        2.05266757f, 2.05385358f, 2.05504027f, 2.05622765f, 2.05741572f,
        2.05860447f, 2.05979391f, 2.06098404f, 2.06217486f, 2.06336636f,
        2.06455855f, 2.06575143f, 2.06694500f, 2.06813926f, 2.06933421f,
        2.07052985f, 2.07172618f, 2.07292320f, 2.07412091f, 2.07531932f,
        2.07651842f, 2.07771821f, 2.07891869f, 2.08011987f, 2.08132174f,
        2.08252431f, 2.08372757f, 2.08493152f, 2.08613617f, 2.08734152f,
        2.08854756f, 2.08975431f, 2.09096174f, 2.09216988f, 2.09337871f,
        2.09458825f, 2.09579848f, 2.09700941f, 2.09822104f, 2.09943337f,
        2.10064640f, 2.10186013f, 2.10307456f, 2.10428970f, 2.10550553f,
        2.10672207f, 2.10793931f, 2.10915726f, 2.11037591f, 2.11159526f,
        2.11281532f, 2.11403608f, 2.11525755f, 2.11647972f, 2.11770260f,
    },
    {
        2.11892619f, 2.12015048f, 2.12137548f, 2.12260119f, 2.12382761f,
        2.12505473f, 2.12628257f, 2.12751111f, 2.12874036f, 2.12997033f,
        2.13120100f, 2.13243239f, 2.13366449f, 2.13489729f, 2.13613082f,
        2.13736505f, 2.13860000f, 2.13983566f, 2.14107203f, 2.14230912f,
        2.14354693f, 2.14478544f, 2.14602468f, 2.14726463f, 2.14850530f,
        2.14974668f, 2.15098878f, 2.15223160f, 2.15347514f, 2.15471939f,
        2.15596437f, 2.15721006f, 2.15845647f, 2.15970361f, 2.16095146f,
        2.16220004f, 2.16344933f, 2.16469935f, 2.16595009f, 2.16720155f,
        2.16845374f, 2.16970665f, 2.17096028f, 2.17221464f, 2.17346973f,
        2.17472553f, 2.17598207f, 2.17723933f, 2.17849731f, 2.17975603f,
        2.18101547f, 2.18227563f, 2.18353653f, 2.18479815f, 2.18606051f,
        2.18732359f, 2.18858740f, 2.18985195f, 2.19111722f, 2.19238322f,
        2.19364996f, 2.19491743f, 2.19618563f, 2.19745456f, 2.19872423f,
        2.19999463f, 2.20126576f, 2.20253763f, 2.20381023f, 2.20508357f,
        2.20635764f, 2.20763245f, 2.20890800f, 2.21018429f, 2.21146131f,
        2.21273907f, 2.21401756f, 2.21529680f, 2.21657677f, 2.21785749f,
        2.21913894f, 2.22042114f, 2.22170408f, 2.22298775f, 2.22427217f,
        2.22555733f, 2.22684324f, 2.22812988f, 2.22941727f, 2.23070541f,
        2.23199429f, 2.23328391f, 2.23457428f, 2.23586539f, 2.23715725f,
        2.23844986f, 2.23974321f, 2.24103731f, 2.24233216f, 2.24362775f,
    },
    {
        2.24492410f, 2.24622119f, 2.24751903f, 2.24881763f, 2.25011697f,
        2.25141706f, 2.25271791f, 2.25401951f, 2.25532185f, 2.25662496f,
        2.25792881f, 2.25923342f, 2.26053878f, 2.26184489f, 2.26315176f,
        2.26445939f, 2.26576777f, 2.26707691f, 2.26838680f, 2.26969745f,
        2.27100886f, 2.27232102f, 2.27363395f, 2.27494763f, 2.27626207f,
        2.27757727f, 2.27889323f, 2.28020995f, 2.28152743f, 2.28284567f,
        2.28416468f, 2.28548445f, 2.28680497f, 2.28812627f, 2.28944832f,
        2.29077114f, 2.29209472f, 2.29341907f, 2.29474419f, 2.29607006f,
        2.29739671f, 2.29872412f, 2.30005230f, 2.30138125f, 2.30271096f,
        2.30404144f, 2.30537269f, 2.30670471f, 2.30803750f, 2.30937106f,
        2.31070539f, 2.31204050f, 2.31337637f, 2.31471301f, 2.31605043f,
        2.31738862f, 2.31872758f, 2.32006732f, 2.32140783f, 2.32274911f,
        2.32409117f, 2.32543401f, 2.32677762f, 2.32812201f, 2.32946717f,
        2.33081311f, 2.33215983f, 2.33350733f, 2.33485561f, 2.33620466f,
        2.33755450f, 2.33890511f, 2.34025651f, 2.34160868f, 2.34296164f,
        2.34431538f, 2.34566990f, 2.34702520f, 2.34838129f, 2.34973816f,
        2.35109581f, 2.35245425f, 2.35381347f, 2.35517348f, 2.35653428f,
        2.35789586f, 2.35925823f, 2.36062138f, 2.36198532f, 2.36335005f,
        2.36471557f, 2.36608188f, 2.36744898f, 2.36881686f, 2.37018554f,
        2.37155501f, 2.37292527f, 2.37429632f, 2.37566816f, 2.37704080f,
    },
    {
        2.37841423f, 2.37978845f, 2.38116347f, 2.38253928f, 2.38391589f,
        2.38529329f, 2.38667149f, 2.38805048f, 2.38943027f, 2.39081086f,
        2.39219224f, 2.39357443f, 2.39495741f, 2.39634119f, 2.39772577f,
        2.39911115f, 2.40049733f, 2.40188432f, 2.40327210f, 2.40466068f,
        2.40605007f, 2.40744026f, 2.40883126f, 2.41022305f, 2.41161566f,
        2.41300906f, 2.41440327f, 2.41579829f, 2.41719411f, 2.41859074f,
        2.41998818f, 2.42138642f, 2.42278547f, 2.42418533f, 2.42558600f,
        2.42698748f, 2.42838977f, 2.42979287f, 2.43119678f, 2.43260150f,
        2.43400703f, 2.43541337f, 2.43682053f, 2.43822850f, 2.43963728f,
        2.44104688f, 2.44245729f, 2.44386851f, 2.44528056f, 2.44669341f,
        2.44810709f, 2.44952158f, 2.45093689f, 2.45235301f, 2.45376995f,
        2.45518772f, 2.45660630f, 2.45802570f, 2.45944592f, 2.46086696f,
        2.46228883f, 2.46371151f, 2.46513502f, 2.46655935f, 2.46798450f,
        2.46941047f, 2.47083727f, 2.47226490f, 2.47369335f, 2.47512262f,
        2.47655272f, 2.47798365f, 2.47941540f, 2.48084798f, 2.48228139f,
        2.48371562f, 2.48515069f, 2.48658658f, 2.48802331f, 2.48946086f,
        2.49089924f, 2.49233846f, 2.49377851f, 2.49521939f, 2.49666110f,
        2.49810364f, 2.49954702f, 2.50099123f, 2.50243628f, 2.50388216f,
        2.50532888f, 2.50677643f, 2.50822482f, 2.50967405f, 2.51112411f,
        2.51257501f, 2.51402675f, 2.51547933f, 2.51693274f, 2.51838700f,
    },
    {
        2.51984210f, 2.52129804f, 2.52275482f, 2.52421244f, 2.52567090f,
        2.52713021f, 2.52859036f, 2.53005135f, 2.53151319f, 2.53297587f,
        2.53443940f, 2.53590377f, 2.53736899f, 2.53883505f, 2.54030197f,
        2.54176972f, 2.54323833f, 2.54470779f, 2.54617809f, 2.54764925f,
        2.54912125f, 2.55059411f, 2.55206782f, 2.55354237f, 2.55501778f,
        2.55649405f, 2.55797116f, 2.55944913f, 2.56092795f, 2.56240763f,
        2.56388816f, 2.56536955f, 2.56685180f, 2.56833490f, 2.56981885f,
        2.57130367f, 2.57278934f, 2.57427587f, 2.57576326f, 2.57725151f,
        2.57874062f, 2.58023059f, 2.58172142f, 2.58321311f, 2.58470566f,
        2.58619908f, 2.58769336f, 2.58918850f, 2.59068450f, 2.59218137f,
        2.59367911f, 2.59517771f, 2.59667718f, 2.59817751f, 2.59967871f,
        2.60118078f, 2.60268371f, 2.60418751f, 2.60569219f, 2.60719773f,
        2.60870414f, 2.61021142f, 2.61171957f, 2.61322860f, 2.61473849f,
        2.61624926f, 2.61776090f, 2.61927342f, 2.62078681f, 2.62230107f,
        2.62381621f, 2.62533222f, 2.62684911f, 2.62836688f, 2.62988552f,
        2.63140504f, 2.63292544f, 2.63444672f, 2.63596887f, 2.63749191f,
        2.63901582f, 2.64054062f, 2.64206629f, 2.64359285f, 2.64512029f,
        2.64664861f, 2.64817782f, 2.64970791f, 2.65123888f, 2.65277074f,
        2.65430348f, 2.65583711f, 2.65737163f, 2.65890703f, 2.66044332f,
        2.66198049f, 2.66351856f, 2.66505751f, 2.66659735f, 2.66813809f,
    },
    {
        2.66967971f, 2.67122222f, 2.67276563f, 2.67430992f, 2.67585511f,
        2.67740119f, 2.67894816f, 2.68049603f, 2.68204480f, 2.68359445f,
        2.68514501f, 2.68669645f, 2.68824880f, 2.68980204f, 2.69135618f,
        2.69291122f, 2.69446715f, 2.69602399f, 2.69758172f, 2.69914036f,
        2.70069989f, 2.70226033f, 2.70382167f, 2.70538391f, 2.70694705f,
        2.70851109f, 2.71007604f, 2.71164190f, 2.71320865f, 2.71477632f,
        2.71634489f, 2.71791436f, 2.71948475f, 2.72105604f, 2.72262823f,
        2.72420134f, 2.72577535f, 2.72735028f, 2.72892611f, 2.73050286f,
        2.73208051f, 2.73365908f, 2.73523856f, 2.73681895f, 2.73840026f,
        2.73998248f, 2.74156561f, 2.74314966f, 2.74473462f, 2.74632050f,
        2.74790729f, 2.74949501f, 2.75108364f, 2.75267318f, 2.75426365f,
        2.75585503f, 2.75744734f, 2.75904056f, 2.76063471f, 2.76222977f,
        2.76382576f, 2.76542267f, 2.76702050f, 2.76861926f, 2.77021894f,
        2.77181954f, 2.77342107f, 2.77502352f, 2.77662690f, 2.77823121f,
        2.77983644f, 2.78144260f, 2.78304969f, 2.78465771f, 2.78626665f,
        2.78787653f, 2.78948733f, 2.79109907f, 2.79271174f, 2.79432534f,
        2.79593987f, 2.79755533f, 2.79917173f, 2.80078906f, 2.80240733f,
        2.80402653f, 2.80564667f, 2.80726774f, 2.80888975f, 2.81051270f,
        2.81213658f, 2.81376141f, 2.81538717f, 2.81701387f, 2.81864151f,
        2.82027009f, 2.82189961f, 2.82353008f, 2.82516148f, 2.82679383f,
    },
    {
        2.82842712f, 2.83006136f, 2.83169654f, 2.83333266f, 2.83496973f,
        2.83660775f, 2.83824671f, 2.83988662f, 2.84152748f, 2.84316928f,
        2.84481204f, 2.84645574f, 2.84810039f, 2.84974599f, 2.85139255f,
        2.85304005f, 2.85468851f, 2.85633792f, 2.85798828f, 2.85963960f,
        2.86129186f, 2.86294509f, 2.86459927f, 2.86625440f, 2.86791050f,
        2.86956754f, 2.87122555f, 2.87288451f, 2.87454444f, 2.87620532f,
        2.87786716f, 2.87952996f, 2.88119372f, 2.88285845f, 2.88452413f,
        2.88619078f, 2.88785839f, 2.88952697f, 2.89119650f, 2.89286701f,
        2.89453847f, 2.89621091f, 2.89788431f, 2.89955868f, 2.90123401f,
        2.90291031f, 2.90458758f, 2.90626582f, 2.90794503f, 2.90962521f,
        2.91130637f, 2.91298849f, 2.91467158f, 2.91635565f, 2.91804069f,
        2.91972670f, 2.92141369f, 2.92310165f, 2.92479059f, 2.92648050f,
        2.92817139f, 2.92986326f, 2.93155610f, 2.93324993f, 2.93494473f,
        2.93664051f, 2.93833727f, 2.94003501f, 2.94173373f, 2.94343343f,
        2.94513412f, 2.94683578f, 2.94853843f, 2.95024207f, 2.95194669f,
        2.95365229f, 2.95535888f, 2.95706646f, 2.95877502f, 2.96048457f,
        2.96219510f, 2.96390663f, 2.96561914f, 2.96733265f, 2.96904714f,
        2.97076263f, 2.97247910f, 2.97419657f, 2.97591503f, 2.97763448f,
        2.97935493f, 2.98107637f, 2.98279880f, 2.98452223f, 2.98624666f,
        2.98797208f, 2.98969850f, 2.99142591f, 2.99315433f, 2.99488374f,
    },
    {
        2.99661415f, 2.99834557f, 3.00007798f, 3.00181139f, 3.00354581f,
        3.00528122f, 3.00701764f, 3.00875507f, 3.01049349f, 3.01223293f,
        3.01397336f, 3.01571481f, 3.01745725f, 3.01920071f, 3.02094517f,
        3.02269064f, 3.02443712f, 3.02618461f, 3.02793311f, 3.02968261f,
        3.03143313f, 3.03318466f, 3.03493721f, 3.03669076f, 3.03844533f,
        3.04020091f, 3.04195751f, 3.04371512f, 3.04547374f, 3.04723339f,
        3.04899405f, 3.05075572f, 3.05251842f, 3.05428213f, 3.05604686f,
        3.05781262f, 3.05957939f, 3.06134718f, 3.06311599f, 3.06488583f,
        3.06665669f, 3.06842857f, 3.07020148f, 3.07197541f, 3.07375036f,
        3.07552634f, 3.07730335f, 3.07908138f, 3.08086044f, 3.08264053f,
        3.08442165f, 3.08620380f, 3.08798697f, 3.08977118f, 3.09155642f,
        3.09334269f, 3.09512999f, 3.09691832f, 3.09870769f, 3.10049809f,
        3.10228952f, 3.10408199f, 3.10587550f, 3.10767004f, 3.10946562f,
        3.11126224f, 3.11305989f, 3.11485859f, 3.11665832f, 3.11845909f,
        3.12026090f, 3.12206376f, 3.12386765f, 3.12567259f, 3.12747857f,
        3.12928560f, 3.13109367f, 3.13290278f, 3.13471294f, 3.13652414f,
        3.13833639f, 3.14014969f, 3.14196403f, 3.14377943f, 3.14559587f,
        3.14741336f, 3.14923191f, 3.15105150f, 3.15287214f, 3.15469384f,
        3.15651659f, 3.15834039f, 3.16016525f, 3.16199116f, 3.16381812f,
        3.16564614f, 3.16747522f, 3.16930536f, 3.17113655f, 3.17296880f,
    },
    {
        3.17480210f, 3.17663647f, 3.17847190f, 3.18030839f, 3.18214594f,
        3.18398455f, 3.18582422f, 3.18766495f, 3.18950675f, 3.19134962f,
        3.19319355f, 3.19503854f, 3.19688460f, 3.19873173f, 3.20057992f,
        3.20242918f, 3.20427951f, 3.20613091f, 3.20798338f, 3.20983692f,
        3.21169153f, 3.21354721f, 3.21540396f, 3.21726179f, 3.21912069f,
        3.22098066f, 3.22284171f, 3.22470384f, 3.22656704f, 3.22843131f,
        3.23029667f, 3.23216310f, 3.23403061f, 3.23589920f, 3.23776887f,
        3.23963962f, 3.24151144f, 3.24338436f, 3.24525835f, 3.24713343f,
        3.24900959f, 3.25088683f, 3.25276516f, 3.25464457f, 3.25652507f,
        3.25840666f, 3.26028933f, 3.26217309f, 3.26405794f, 3.26594388f,
        3.26783091f, 3.26971902f, 3.27160823f, 3.27349854f, 3.27538993f,
        3.27728241f, 3.27917599f, 3.28107067f, 3.28296644f, 3.28486330f,
        3.28676126f, 3.28866031f, 3.29056047f, 3.29246172f, 3.29436407f,
        3.29626752f, 3.29817207f, 3.30007772f, 3.30198447f, 3.30389232f,
        3.30580127f, 3.30771133f, 3.30962249f, 3.31153476f, 3.31344813f,
        3.31536260f, 3.31727818f, 3.31919487f, 3.32111267f, 3.32303157f,
        3.32495158f, 3.32687271f, 3.32879494f, 3.33071828f, 3.33264274f,
        3.33456830f, 3.33649498f, 3.33842277f, 3.34035168f, 3.34228170f,
        3.34421283f, 3.34614508f, 3.34807845f, 3.35001294f, 3.35194854f,
        3.35388526f, 3.35582310f, 3.35776206f, 3.35970214f, 3.36164334f,
    },
    {
        3.36358566f, 3.36552911f, 3.36747367f, 3.36941936f, 3.37136618f,
        3.37331412f, 3.37526318f, 3.37721338f, 3.37916469f, 3.38111714f,
        3.38307071f, 3.38502542f, 3.38698125f, 3.38893821f, 3.39089630f,
        3.39285553f, 3.39481589f, 3.39677737f, 3.39874000f, 3.40070375f,
        3.40266864f, 3.40463467f, 3.40660183f, 3.40857013f, 3.41053957f,
        3.41251014f, 3.41448185f, 3.41645471f, 3.41842870f, 3.42040383f,
        3.42238010f, 3.42435752f, 3.42633608f, 3.42831578f, 3.43029662f,
        3.43227861f, 3.43426175f, 3.43624603f, 3.43823145f, 3.44021803f,
        3.44220575f, 3.44419462f, 3.44618464f, 3.44817581f, 3.45016813f,
        3.45216160f, 3.45415622f, 3.45615200f, 3.45814893f, 3.46014701f,
        3.46214624f, 3.46414664f, 3.46614818f, 3.46815089f, 3.47015475f,
        3.47215977f, 3.47416595f, 3.47617328f, 3.47818178f, 3.48019144f,
        3.48220225f, 3.48421423f, 3.48622738f, 3.48824168f, 3.49025715f,
        3.49227378f, 3.49429158f, 3.49631055f, 3.49833068f, 3.50035198f,
        3.50237445f, 3.50439808f, 3.50642289f, 3.50844886f, 3.51047600f,
        3.51250432f, 3.51453381f, 3.51656447f, 3.51859630f, 3.52062931f,
        3.52266349f, 3.52469885f, 3.52673539f, 3.52877310f, 3.53081199f,
        3.53285205f, 3.53489330f, 3.53693572f, 3.53897932f, 3.54102411f,
        3.54307008f, 3.54511723f, 3.54716556f, 3.54921507f, 3.55126577f,
        3.55331766f, 3.55537072f, 3.55742498f, 3.55948042f, 3.56153705f,
    },
    {
        3.56359487f, 3.56565388f, 3.56771408f, 3.56977547f, 3.57183804f,
        3.57390181f, 3.57596678f, 3.57803293f, 3.58010028f, 3.58216883f,
        3.58423857f, 3.58630950f, 3.58838164f, 3.59045496f, 3.59252949f,
        3.59460522f, 3.59668214f, 3.59876027f, 3.60083959f, 3.60292012f,
        3.60500185f, 3.60708478f, 3.60916892f, 3.61125426f, 3.61334080f,
        3.61542855f, 3.61751751f, 3.61960767f, 3.62169905f, 3.62379162f,
        3.62588541f, 3.62798041f, 3.63007662f, 3.63217404f, 3.63427267f,
        3.63637252f, 3.63847358f, 3.64057585f, 3.64267933f, 3.64478404f,
        3.64688995f, 3.64899709f, 3.65110544f, 3.65321501f, 3.65532580f,
        3.65743781f, 3.65955104f, 3.66166549f, 3.66378116f, 3.66589806f,
        3.66801617f, 3.67013551f, 3.67225608f, 3.67437787f, 3.67650089f,
        3.67862513f, 3.68075060f, 3.68287730f, 3.68500523f, 3.68713439f,
        3.68926477f, 3.69139639f, 3.69352924f, 3.69566333f, 3.69779864f,
        3.69993519f, 3.70207297f, 3.70421199f, 3.70635225f, 3.70849374f,
        3.71063647f, 3.71278043f, 3.71492564f, 3.71707209f, 3.71921977f,
        3.72136870f, 3.72351886f, 3.72567027f, 3.72782293f, 3.72997682f,
        3.73213197f, 3.73428835f, 3.73644599f, 3.73860486f, 3.74076499f,
        3.74292637f, 3.74508899f, 3.74725286f, 3.74941799f, 3.75158436f,
        3.75375199f, 3.75592087f, 3.75809100f, 3.76026238f, 3.76243502f,
        3.76460892f, 3.76678407f, 3.76896048f, 3.77113814f, 3.77331707f,
    },
    {
        3.77549725f, 3.77767869f, 3.77986140f, 3.78204536f, 3.78423059f,
        3.78641708f, 3.78860483f, 3.79079384f, 3.79298412f, 3.79517567f,
        3.79736848f, 3.79956256f, 3.80175791f, 3.80395453f, 3.80615241f,
        3.80835157f, 3.81055199f, 3.81275369f, 3.81495666f, 3.81716090f,
        3.81936642f, 3.82157321f, 3.82378127f, 3.82599061f, 3.82820123f,
        3.83041312f, 3.83262630f, 3.83484075f, 3.83705648f, 3.83927349f,
        3.84149178f, 3.84371135f, 3.84593221f, 3.84815435f, 3.85037777f,
        3.85260248f, 3.85482847f, 3.85705575f, 3.85928432f, 3.86151417f,
        3.86374532f, 3.86597775f, 3.86821147f, 3.87044648f, 3.87268278f,
        3.87492038f, 3.87715927f, 3.87939945f, 3.88164093f, 3.88388370f,
        3.88612776f, 3.88837313f, 3.89061979f, 3.89286775f, 3.89511701f,
        3.89736756f, 3.89961942f, 3.90187258f, 3.90412704f, 3.90638281f,
        3.90863987f, 3.91089824f, 3.91315792f, 3.91541890f, 3.91768119f,
        3.91994479f, 3.92220969f, 3.92447590f, 3.92674342f, 3.92901225f,
        3.93128239f, 3.93355385f, 3.93582661f, 3.93810069f, 3.94037609f,
        3.94265279f, 3.94493082f, 3.94721016f, 3.94949081f, 3.95177279f,
        3.95405608f, 3.95634069f, 3.95862663f, 3.96091388f, 3.96320245f,
        3.96549235f, 3.96778357f, 3.97007611f, 3.97236998f, 3.97466518f,
        3.97696170f, 3.97925954f, 3.98155872f, 3.98385922f, 3.98616105f,
        3.98846421f, 3.99076871f, 3.99307453f, 3.99538169f, 3.99769018f,
    },
    {
        4.00000000f, 4.00231116f, 4.00462365f, 4.00693748f, 4.00925265f,
        4.01156915f, 4.01388699f, 4.01620618f, 4.01852670f, 4.02084856f,
        4.02317176f, 4.02549631f, 4.02782220f, 4.03014943f, 4.03247801f,
        4.03480794f, 4.03713920f, 4.03947182f, 4.04180579f, 4.04414110f,
        4.04647776f, 4.04881577f, 4.05115514f, 4.05349585f, 4.05583792f,
        4.05818134f, 4.06052611f, 4.06287224f, 4.06521973f, 4.06756857f,
        4.06991877f, 4.07227032f, 4.07462324f, 4.07697751f, 4.07933315f,
        4.08169015f, 4.08404850f, 4.08640822f, 4.08876931f, 4.09113175f,
        4.09349557f, 4.09586075f, 4.09822729f, 4.10059520f, 4.10296449f,
        4.10533514f, 4.10770716f, 4.11008055f, 4.11245531f, 4.11483144f,
        4.11720895f, 4.11958783f, 4.12196808f, 4.12434971f, 4.12673272f,
        4.12911710f, 4.13150286f, 4.13389000f, 4.13627852f, 4.13866842f,
        4.14105970f, 4.14345236f, 4.14584640f, 4.14824183f, 4.15063864f,
        4.15303683f, 4.15543641f, 4.15783738f, 4.16023974f, 4.16264348f,
        4.16504861f, 4.16745513f, 4.16986304f, 4.17227235f, 4.17468304f,
        4.17709513f, 4.17950861f, 4.18192349f, 4.18433976f, 4.18675743f,
        4.18917649f, 4.19159695f, 4.19401881f, 4.19644207f, 4.19886673f,
        4.20129280f, 4.20372026f, 4.20614912f, 4.20857939f, 4.21101107f,
        4.21344414f, 4.21587863f, 4.21831452f, 4.22075182f, 4.22319052f,
        4.22563064f, 4.22807216f, 4.23051510f, 4.23295945f, 4.23540520f,
    },
    {
        4.23785238f, 4.24030096f, 4.24275097f, 4.24520238f, 4.24765522f,
        4.25010947f, 4.25256514f, 4.25502222f, 4.25748073f, 4.25994066f,
        4.26240201f, 4.26486478f, 4.26732897f, 4.26979459f, 4.27226163f,
        4.27473010f, 4.27719999f, 4.27967132f, 4.28214406f, 4.28461824f,
        4.28709385f, 4.28957089f, 4.29204936f, 4.29452926f, 4.29701059f,
        4.29949336f, 4.30197756f, 4.30446320f, 4.30695027f, 4.30943878f,
        4.31192873f, 4.31442012f, 4.31691295f, 4.31940721f, 4.32190292f,
        4.32440007f, 4.32689866f, 4.32939870f, 4.33190018f, 4.33440311f,
        4.33690748f, 4.33941330f, 4.34192057f, 4.34442928f, 4.34693945f,
        4.34945107f, 4.35196413f, 4.35447865f, 4.35699462f, 4.35951205f,
        4.36203093f, 4.36455127f, 4.36707306f, 4.36959631f, 4.37212101f,
        4.37464718f, 4.37717481f, 4.37970389f, 4.38223444f, 4.38476645f,
        4.38729992f, 4.38983485f, 4.39237126f, 4.39490912f, 4.39744845f,
        4.39998925f, 4.40253152f, 4.40507526f, 4.40762046f, 4.41016714f,
        4.41271529f, 4.41526491f, 4.41781600f, 4.42036857f, 4.42292261f,
        4.42547813f, 4.42803513f, 4.43059360f, 4.43315355f, 4.43571498f,
        4.43827789f, 4.44084228f, 4.44340815f, 4.44597551f, 4.44854434f,
        4.45111467f, 4.45368647f, 4.45625977f, 4.45883455f, 4.46141081f,
        4.46398857f, 4.46656782f, 4.46914855f, 4.47173078f, 4.47431450f,
        4.47689971f, 4.47948642f, 4.48207462f, 4.48466431f, 4.48725550f,
    },
    {
        4.48984819f, 4.49244238f, 4.49503807f, 4.49763525f, 4.50023394f,
        4.50283413f, 4.50543582f, 4.50803901f, 4.51064371f, 4.51324991f,
        4.51585762f, 4.51846683f, 4.52107756f, 4.52368979f, 4.52630353f,
        4.52891878f, 4.53153554f, 4.53415381f, 4.53677360f, 4.53939490f,
        4.54201772f, 4.54464205f, 4.54726789f, 4.54989526f, 4.55252414f,
        4.55515454f, 4.55778646f, 4.56041990f, 4.56305486f, 4.56569135f,
        4.56832936f, 4.57096889f, 4.57360995f, 4.57625253f, 4.57889664f,
        4.58154228f, 4.58418945f, 4.58683814f, 4.58948837f, 4.59214013f,
        4.59479342f, 4.59744824f, 4.60010460f, 4.60276249f, 4.60542192f,
        4.60808289f, 4.61074539f, 4.61340943f, 4.61607501f, 4.61874213f,
        4.62141079f, 4.62408099f, 4.62675274f, 4.62942602f, 4.63210086f,
        4.63477724f, 4.63745516f, 4.64013464f, 4.64281566f, 4.64549823f,
        4.64818235f, 4.65086802f, 4.65355524f, 4.65624402f, 4.65893435f,
        4.66162623f, 4.66431967f, 4.66701466f, 4.66971122f, 4.67240933f,
        4.67510899f, 4.67781022f, 4.68051301f, 4.68321736f, 4.68592328f,
        4.68863076f, 4.69133980f, 4.69405040f, 4.69676258f, 4.69947632f,
        4.70219163f, 4.70490850f, 4.70762695f, 4.71034697f, 4.71306856f,
        4.71579172f, 4.71851645f, 4.72124276f, 4.72397065f, 4.72670011f,
        4.72943114f, 4.73216376f, 4.73489795f, 4.73763373f, 4.74037108f,
        4.74311002f, 4.74585054f, 4.74859264f, 4.75133633f, 4.75408160f,
    },
    {
        4.75682846f, 4.75957691f, 4.76232694f, 4.76507856f, 4.76783177f,
        4.77058658f, 4.77334297f, 4.77610096f, 4.77886054f, 4.78162172f,
        4.78438449f, 4.78714885f, 4.78991482f, 4.79268238f, 4.79545154f,
        4.79822230f, 4.80099467f, 4.80376863f, 4.80654420f, 4.80932137f,
        4.81210014f, 4.81488053f, 4.81766251f, 4.82044611f, 4.82323131f,
        4.82601812f, 4.82880655f, 4.83159658f, 4.83438823f, 4.83718148f,
        4.83997636f, 4.84277284f, 4.84557095f, 4.84837067f, 4.85117201f,
        4.85397496f, 4.85677954f, 4.85958573f, 4.86239355f, 4.86520299f,
        4.86801405f, 4.87082674f, 4.87364105f, 4.87645699f, 4.87927456f,
        4.88209375f, 4.88491458f, 4.88773703f, 4.89056111f, 4.89338683f,
        4.89621417f, 4.89904315f, 4.90187377f, 4.90470602f, 4.90753991f,
        4.91037543f, 4.91321260f, 4.91605140f, 4.91889184f, 4.92173393f,
        4.92457765f, 4.92742302f, 4.93027004f, 4.93311869f, 4.93596900f,
        4.93882095f, 4.94167455f, 4.94452980f, 4.94738669f, 4.95024524f,
        4.95310544f, 4.95596729f, 4.95883080f, 4.96169596f, 4.96456278f,
        4.96743125f, 4.97030138f, 4.97317317f, 4.97604661f, 4.97892172f,
        4.98179849f, 4.98467692f, 4.98755701f, 4.99043877f, 4.99332220f,
        4.99620728f, 4.99909404f, 5.00198246f, 5.00487256f, 5.00776432f,
        5.01065775f, 5.01355286f, 5.01644964f, 5.01934809f, 5.02224822f,
        5.02515002f, 5.02805350f, 5.03095865f, 5.03386549f, 5.03677400f,
    },
    {
        5.03968420f, 5.04259608f, 5.04550964f, 5.04842488f, 5.05134180f,
        5.05426042f, 5.05718072f, 5.06010270f, 5.06302638f, 5.06595174f,
        5.06887879f, 5.07180754f, 5.07473798f, 5.07767011f, 5.08060393f,
        5.08353945f, 5.08647667f, 5.08941558f, 5.09235619f, 5.09529850f,
        5.09824251f, 5.10118822f, 5.10413563f, 5.10708475f, 5.11003557f,
        5.11298809f, 5.11594233f, 5.11889826f, 5.12185591f, 5.12481526f,
        5.12777633f, 5.13073910f, 5.13370359f, 5.13666979f, 5.13963770f,
        5.14260733f, 5.14557868f, 5.14855174f, 5.15152652f, 5.15450302f,
        5.15748123f, 5.16046117f, 5.16344283f, 5.16642622f, 5.16941132f,
        5.17239815f, 5.17538671f, 5.17837700f, 5.18136901f, 5.18436275f,
        5.18735822f, 5.19035542f, 5.19335435f, 5.19635502f, 5.19935742f,
        5.20236155f, 5.20536742f, 5.20837503f, 5.21138437f, 5.21439546f,
        5.21740828f, 5.22042284f, 5.22343915f, 5.22645720f, 5.22947699f,
        5.23249853f, 5.23552181f, 5.23854684f, 5.24157362f, 5.24460214f,
        5.24763242f, 5.25066444f, 5.25369822f, 5.25673376f, 5.25977104f,
        5.26281008f, 5.26585088f, 5.26889343f, 5.27193774f, 5.27498381f,
        5.27803164f, 5.28108123f, 5.28413259f, 5.28718570f, 5.29024058f,
        5.29329723f, 5.29635564f, 5.29941582f, 5.30247777f, 5.30554148f,
        5.30860697f, 5.31167423f, 5.31474326f, 5.31781406f, 5.32088664f,
        5.32396099f, 5.32703712f, 5.33011502f, 5.33319471f, 5.33627617f,
    },
    {
        5.33935942f, 5.34244444f, 5.34553125f, 5.34861984f, 5.35171022f,
        5.35480238f, 5.35789633f, 5.36099207f, 5.36408959f, 5.36718891f,
        5.37029001f, 5.37339291f, 5.37649760f, 5.37960408f, 5.38271236f,
        5.38582244f, 5.38893431f, 5.39204798f, 5.39516345f, 5.39828071f,
        5.40139978f, 5.40452066f, 5.40764333f, 5.41076781f, 5.41389410f,
        5.41702219f, 5.42015209f, 5.42328379f, 5.42641731f, 5.42955264f,
        5.43268978f, 5.43582873f, 5.43896949f, 5.44211207f, 5.44525647f,
        5.44840268f, 5.45155071f, 5.45470056f, 5.45785223f, 5.46100572f,
        5.46416103f, 5.46731816f, 5.47047712f, 5.47363791f, 5.47680052f,
        5.47996495f, 5.48313122f, 5.48629932f, 5.48946924f, 5.49264100f,
        5.49581459f, 5.49899001f, 5.50216727f, 5.50534637f, 5.50852730f,
        5.51171007f, 5.51489468f, 5.51808112f, 5.52126941f, 5.52445955f,
        5.52765152f, 5.53084534f, 5.53404100f, 5.53723851f, 5.54043787f,
        5.54363908f, 5.54684214f, 5.55004704f, 5.55325380f, 5.55646241f,
        5.55967288f, 5.56288520f, 5.56609938f, 5.56931541f, 5.57253330f,
        5.57575305f, 5.57897467f, 5.58219814f, 5.58542347f, 5.58865067f,
        5.59187974f, 5.59511067f, 5.59834346f, 5.60157813f, 5.60481466f,
        5.60805306f, 5.61129334f, 5.61453548f, 5.61777950f, 5.62102540f,
        5.62427317f, 5.62752281f, 5.63077434f, 5.63402774f, 5.63728302f,
        5.64054018f, 5.64379923f, 5.64706016f, 5.65032297f, 5.65358767f,
    },
    {
        5.65685425f, 5.66012272f, 5.66339308f, 5.66666533f, 5.66993947f,
        5.67321550f, 5.67649342f, 5.67977324f, 5.68305496f, 5.68633857f,
        5.68962407f, 5.69291148f, 5.69620078f, 5.69949199f, 5.70278509f,
        5.70608010f, 5.70937702f, 5.71267583f, 5.71597656f, 5.71927919f,
        5.72258373f, 5.72589018f, 5.72919854f, 5.73250881f, 5.73582099f,
        5.73913509f, 5.74245110f, 5.74576903f, 5.74908887f, 5.75241064f,
        5.75573432f, 5.75905992f, 5.76238745f, 5.76571689f, 5.76904827f,
        5.77238156f, 5.77571678f, 5.77905393f, 5.78239301f, 5.78573401f,
        5.78907695f, 5.79242182f, 5.79576862f, 5.79911735f, 5.80246802f,
        5.80582063f, 5.80917517f, 5.81253165f, 5.81589007f, 5.81925043f,
        5.82261273f, 5.82597698f, 5.82934316f, 5.83271130f, 5.83608138f,
        5.83945340f, 5.84282738f, 5.84620330f, 5.84958118f, 5.85296100f,
        5.85634278f, 5.85972652f, 5.86311221f, 5.86649985f, 5.86988945f,
        5.87328101f, 5.87667453f, 5.88007001f, 5.88346746f, 5.88686686f,
        5.89026823f, 5.89367157f, 5.89707687f, 5.90048414f, 5.90389338f,
        5.90730458f, 5.91071776f, 5.91413291f, 5.91755004f, 5.92096914f,
        5.92439021f, 5.92781326f, 5.93123829f, 5.93466530f, 5.93809428f,
        5.94152525f, 5.94495820f, 5.94839314f, 5.95183006f, 5.95526896f,
        5.95870985f, 5.96215273f, 5.96559760f, 5.96904446f, 5.97249331f,
        5.97594416f, 5.97939699f, 5.98285183f, 5.98630866f, 5.98976748f,
    },
    {
        5.99322831f, 5.99669113f, 6.00015596f, 6.00362278f, 6.00709161f,
        6.01056245f, 6.01403529f, 6.01751014f, 6.02098699f, 6.02446585f,
        6.02794673f, 6.03142961f, 6.03491451f, 6.03840142f, 6.04189034f,
        6.04538128f, 6.04887424f, 6.05236922f, 6.05586621f, 6.05936523f,
        6.06286627f, 6.06636933f, 6.06987441f, 6.07338152f, 6.07689066f,
        6.08040182f, 6.08391501f, 6.08743024f, 6.09094749f, 6.09446677f,
        6.09798809f, 6.10151145f, 6.10503684f, 6.10856426f, 6.11209373f,
        6.11562523f, 6.11915877f, 6.12269436f, 6.12623199f, 6.12977166f,
        6.13331338f, 6.13685714f, 6.14040295f, 6.14395081f, 6.14750073f,
        6.15105269f, 6.15460670f, 6.15816277f, 6.16172089f, 6.16528107f,
        6.16884330f, 6.17240759f, 6.17597395f, 6.17954236f, 6.18311283f,
        6.18668537f, 6.19025997f, 6.19383664f, 6.19741538f, 6.20099618f,
        6.20457905f, 6.20816399f, 6.21175100f, 6.21534008f, 6.21893124f,
        6.22252448f, 6.22611979f, 6.22971717f, 6.23331664f, 6.23691818f,
        6.24052181f, 6.24412752f, 6.24773531f, 6.25134518f, 6.25495715f,
        6.25857119f, 6.26218733f, 6.26580556f, 6.26942587f, 6.27304828f,
        6.27667278f, 6.28029938f, 6.28392807f, 6.28755886f, 6.29119174f,
        6.29482673f, 6.29846381f, 6.30210300f, 6.30574429f, 6.30938768f,
        6.31303318f, 6.31668078f, 6.32033049f, 6.32398232f, 6.32763625f,
        6.33129229f, 6.33495044f, 6.33861071f, 6.34227309f, 6.34593759f,
    },
    {
        6.34960421f, 6.35327294f, 6.35694380f, 6.36061677f, 6.36429187f,
        6.36796909f, 6.37164844f, 6.37532991f, 6.37901351f, 6.38269923f,
        6.38638709f, 6.39007708f, 6.39376920f, 6.39746345f, 6.40115984f,
        6.40485836f, 6.40855902f, 6.41226182f, 6.41596676f, 6.41967383f,
        6.42338305f, 6.42709442f, 6.43080793f, 6.43452358f, 6.43824138f,
        6.44196133f, 6.44568343f, 6.44940767f, 6.45313407f, 6.45686263f,
        6.46059333f, 6.46432620f, 6.46806122f, 6.47179840f, 6.47553773f,
        6.47927923f, 6.48302289f, 6.48676871f, 6.49051670f, 6.49426685f,
        6.49801917f, 6.50177366f, 6.50553031f, 6.50928914f, 6.51305014f,
        6.51681331f, 6.52057866f, 6.52434618f, 6.52811588f, 6.53188776f,
        6.53566181f, 6.53943805f, 6.54321647f, 6.54699707f, 6.55077986f,
        6.55456483f, 6.55835199f, 6.56214133f, 6.56593287f, 6.56972660f,
        6.57352252f, 6.57732063f, 6.58112094f, 6.58492344f, 6.58872814f,
        6.59253504f, 6.59634413f, 6.60015543f, 6.60396893f, 6.60778464f,
        6.61160255f, 6.61542266f, 6.61924498f, 6.62306951f, 6.62689625f,
        6.63072520f, 6.63455637f, 6.63838974f, 6.64222534f, 6.64606314f,
        6.64990317f, 6.65374541f, 6.65758988f, 6.66143656f, 6.66528547f,
        6.66913660f, 6.67298996f, 6.67684554f, 6.68070336f, 6.68456340f,
        6.68842567f, 6.69229017f, 6.69615690f, 6.70002587f, 6.70389708f,
        6.70777052f, 6.71164620f, 6.71552412f, 6.71940428f, 6.72328668f,
    },
    {
        6.72717132f, 6.73105821f, 6.73494735f, 6.73883873f, 6.74273236f,
        6.74662824f, 6.75052637f, 6.75442675f, 6.75832939f, 6.76223428f,
        6.76614143f, 6.77005083f, 6.77396250f, 6.77787642f, 6.78179261f,
        6.78571106f, 6.78963177f, 6.79355475f, 6.79747999f, 6.80140751f,
        6.80533729f, 6.80926934f, 6.81320366f, 6.81714026f, 6.82107913f,
        6.82502028f, 6.82896371f, 6.83290941f, 6.83685739f, 6.84080766f,
        6.84476021f, 6.84871504f, 6.85267215f, 6.85663155f, 6.86059324f,
        6.86455722f, 6.86852349f, 6.87249205f, 6.87646291f, 6.88043605f,
        6.88441150f, 6.88838924f, 6.89236928f, 6.89635162f, 6.90033626f,
        6.90432320f, 6.90831244f, 6.91230399f, 6.91629785f, 6.92029401f,
        6.92429249f, 6.92829327f, 6.93229637f, 6.93630177f, 6.94030950f,
        6.94431954f, 6.94833189f, 6.95234656f, 6.95636356f, 6.96038287f,
        6.96440451f, 6.96842847f, 6.97245475f, 6.97648336f, 6.98051430f,
        6.98454757f, 6.98858317f, 6.99262110f, 6.99666136f, 7.00070396f,
        7.00474889f, 7.00879616f, 7.01284577f, 7.01689772f, 7.02095201f,
        7.02500864f, 7.02906762f, 7.03312894f, 7.03719261f, 7.04125862f,
        7.04532699f, 7.04939770f, 7.05347077f, 7.05754619f, 7.06162397f,
        7.06570410f, 7.06978659f, 7.07387144f, 7.07795865f, 7.08204822f,
        7.08614015f, 7.09023445f, 7.09433111f, 7.09843014f, 7.10253154f,
        7.10663531f, 7.11074145f, 7.11484996f, 7.11896085f, 7.12307411f,
    },
    {
        7.12718975f, 7.13130776f, 7.13542816f, 7.13955093f, 7.14367609f,
        7.14780363f, 7.15193356f, 7.15606587f, 7.16020057f, 7.16433766f,
        7.16847714f, 7.17261901f, 7.17676327f, 7.18090993f, 7.18505898f,
        7.18921043f, 7.19336429f, 7.19752054f, 7.20167919f, 7.20584024f,
        7.21000370f, 7.21416957f, 7.21833784f, 7.22250852f, 7.22668161f,
        7.23085711f, 7.23503502f, 7.23921535f, 7.24339809f, 7.24758325f,
        7.25177083f, 7.25596082f, 7.26015324f, 7.26434808f, 7.26854535f,
        7.27274504f, 7.27694715f, 7.28115170f, 7.28535867f, 7.28956807f,
        7.29377991f, 7.29799418f, 7.30221088f, 7.30643002f, 7.31065160f,
        7.31487562f, 7.31910208f, 7.32333098f, 7.32756232f, 7.33179611f,
        7.33603235f, 7.34027103f, 7.34451216f, 7.34875574f, 7.35300178f,
        7.35725026f, 7.36150120f, 7.36575460f, 7.37001046f, 7.37426877f,
        7.37852955f, 7.38279279f, 7.38705849f, 7.39132665f, 7.39559728f,
        7.39987038f, 7.40414595f, 7.40842399f, 7.41270450f, 7.41698748f,
        7.42127294f, 7.42556087f, 7.42985128f, 7.43414417f, 7.43843954f,
        7.44273739f, 7.44703773f, 7.45134055f, 7.45564586f, 7.45995365f,
        7.46426393f, 7.46857671f, 7.47289197f, 7.47720973f, 7.48152998f,
        7.48585273f, 7.49017798f, 7.49450573f, 7.49883597f, 7.50316872f,
        7.50750397f, 7.51184173f, 7.51618199f, 7.52052476f, 7.52487005f,
        7.52921784f, 7.53356814f, 7.53792096f, 7.54227629f, 7.54663414f,
    },
    {
        7.55099450f, 7.55535739f, 7.55972279f, 7.56409072f, 7.56846117f,
        7.57283415f, 7.57720966f, 7.58158769f, 7.58596825f, 7.59035134f,
        7.59473697f, 7.59912513f, 7.60351582f, 7.60790905f, 7.61230482f,
        7.61670313f, 7.62110398f, 7.62550738f, 7.62991332f, 7.63432180f,
        7.63873283f, 7.64314641f, 7.64756254f, 7.65198122f, 7.65640246f,
        7.66082625f, 7.66525259f, 7.66968149f, 7.67411295f, 7.67854698f,
        7.68298356f, 7.68742271f, 7.69186442f, 7.69630870f, 7.70075554f,
        7.70520496f, 7.70965695f, 7.71411151f, 7.71856864f, 7.72302835f,
        7.72749063f, 7.73195549f, 7.73642294f, 7.74089296f, 7.74536557f,
        7.74984076f, 7.75431854f, 7.75879890f, 7.76328185f, 7.76776739f,
        7.77225553f, 7.77674626f, 7.78123958f, 7.78573550f, 7.79023401f,
        7.79473513f, 7.79923885f, 7.80374516f, 7.80825409f, 7.81276561f,
        7.81727975f, 7.82179649f, 7.82631584f, 7.83083780f, 7.83536238f,
        7.83988957f, 7.84441938f, 7.84895180f, 7.85348684f, 7.85802450f,
        7.86256479f, 7.86710770f, 7.87165323f, 7.87620139f, 7.88075217f,
        7.88530559f, 7.88986164f, 7.89442032f, 7.89898163f, 7.90354558f,
        7.90811216f, 7.91268139f, 7.91725325f, 7.92182776f, 7.92640491f,
        7.93098470f, 7.93556714f, 7.94015223f, 7.94473996f, 7.94933035f,
        7.95392339f, 7.95851908f, 7.96311743f, 7.96771844f, 7.97232210f,
        7.97692843f, 7.98153741f, 7.98614906f, 7.99076337f, 7.99538035f,
    },
    {
        8.00000000f, 8.00462232f, 8.00924730f, 8.01387496f, 8.01850529f,
        8.02313830f, 8.02777399f, 8.03241235f, 8.03705340f, 8.04169712f,
        8.04634353f, 8.05099262f, 8.05564440f, 8.06029887f, 8.06495602f,
        8.06961587f, 8.07427841f, 8.07894364f, 8.08361157f, 8.08828220f,
        8.09295552f, 8.09763155f, 8.10231027f, 8.10699170f, 8.11167584f,
        8.11636268f, 8.12105223f, 8.12574449f, 8.13043946f, 8.13513714f,
        8.13983754f, 8.14454065f, 8.14924648f, 8.15395503f, 8.15866630f,
        8.16338029f, 8.16809701f, 8.17281645f, 8.17753861f, 8.18226351f,
        8.18699114f, 8.19172149f, 8.19645458f, 8.20119041f, 8.20592897f,
        8.21067027f, 8.21541431f, 8.22016109f, 8.22491061f, 8.22966288f,
        8.23441789f, 8.23917565f, 8.24393616f, 8.24869942f, 8.25346543f,
        8.25823420f, 8.26300572f, 8.26778000f, 8.27255704f, 8.27733683f,
        8.28211939f, 8.28690471f, 8.29169280f, 8.29648365f, 8.30127727f,
        8.30607366f, 8.31087283f, 8.31567476f, 8.32047947f, 8.32528696f,
        8.33009722f, 8.33491026f, 8.33972609f, 8.34454469f, 8.34936608f,
        8.35419026f, 8.35901722f, 8.36384698f, 8.36867952f, 8.37351485f,
        8.37835298f, 8.38319391f, 8.38803763f, 8.39288415f, 8.39773347f,
        8.40258559f, 8.40744052f, 8.41229825f, 8.41715879f, 8.42202213f,
        8.42688829f, 8.43175726f, 8.43662904f, 8.44150363f, 8.44638104f,
        8.45126127f, 8.45614432f, 8.46103020f, 8.46591889f, 8.47081041f,
    },
    {
        8.47570475f, 8.48060193f, 8.48550193f, 8.49040476f, 8.49531043f,
        8.50021893f, 8.50513027f, 8.51004445f, 8.51496146f, 8.51988132f,
        8.52480401f, 8.52972956f, 8.53465794f, 8.53958918f, 8.54452326f,
        8.54946020f, 8.55439999f, 8.55934263f, 8.56428813f, 8.56923649f,
        8.57418770f, 8.57914178f, 8.58409871f, 8.58905852f, 8.59402118f,
        8.59898672f, 8.60395512f, 8.60892640f, 8.61390055f, 8.61887757f,
        8.62385746f, 8.62884024f, 8.63382589f, 8.63881443f, 8.64380584f,
        8.64880014f, 8.65379733f, 8.65879740f, 8.66380036f, 8.66880622f,
        8.67381496f, 8.67882660f, 8.68384114f, 8.68885857f, 8.69387890f,
        8.69890213f, 8.70392827f, 8.70895730f, 8.71398925f, 8.71902410f,
        8.72406186f, 8.72910253f, 8.73414612f, 8.73919261f, 8.74424203f,
        8.74929436f, 8.75434961f, 8.75940778f, 8.76446888f, 8.76953289f,
        8.77459984f, 8.77966971f, 8.78474251f, 8.78981824f, 8.79489691f,
        8.79997851f, 8.80506304f, 8.81015051f, 8.81524093f, 8.82033428f,
        8.82543058f, 8.83052982f, 8.83563201f, 8.84073714f, 8.84584523f,
        8.85095626f, 8.85607025f, 8.86118720f, 8.86630710f, 8.87142996f,
        8.87655578f, 8.88168456f, 8.88681630f, 8.89195101f, 8.89708869f,
        8.90222933f, 8.90737295f, 8.91251953f, 8.91766909f, 8.92282163f,
        8.92797714f, 8.93313563f, 8.93829710f, 8.94346156f, 8.94862900f,
        8.95379942f, 8.95897283f, 8.96414923f, 8.96932862f, 8.97451101f,
    },
    {
        8.97969639f, 8.98488476f, 8.99007613f, 8.99527050f, 9.00046788f,
        9.00566825f, 9.01087163f, 9.01607802f, 9.02128742f, 9.02649982f,
        9.03171524f, 9.03693367f, 9.04215511f, 9.04737958f, 9.05260706f,
        9.05783756f, 9.06307108f, 9.06830763f, 9.07354720f, 9.07878980f,
        9.08403543f, 9.08928409f, 9.09453579f, 9.09979051f, 9.10504828f,
        9.11030908f, 9.11557292f, 9.12083980f, 9.12610973f, 9.13138270f,
        9.13665871f, 9.14193778f, 9.14721990f, 9.15250506f, 9.15779328f,
        9.16308456f, 9.16837890f, 9.17367629f, 9.17897674f, 9.18428026f,
        9.18958684f, 9.19489649f, 9.20020920f, 9.20552499f, 9.21084384f,
        9.21616577f, 9.22149077f, 9.22681885f, 9.23215001f, 9.23748425f,
        9.24282157f, 9.24816198f, 9.25350547f, 9.25885205f, 9.26420172f,
        9.26955448f, 9.27491033f, 9.28026927f, 9.28563132f, 9.29099646f,
        9.29636470f, 9.30173604f, 9.30711048f, 9.31248803f, 9.31786869f,
        9.32325246f, 9.32863934f, 9.33402933f, 9.33942243f, 9.34481865f,
        9.35021799f, 9.35562045f, 9.36102603f, 9.36643473f, 9.37184656f,
        9.37726151f, 9.38267959f, 9.38810081f, 9.39352515f, 9.39895263f,
        9.40438325f, 9.40981700f, 9.41525390f, 9.42069393f, 9.42613711f,
        9.43158343f, 9.43703290f, 9.44248552f, 9.44794129f, 9.45340021f,
        9.45886229f, 9.46432752f, 9.46979591f, 9.47526746f, 9.48074217f,
        9.48622004f, 9.49170108f, 9.49718528f, 9.50267266f, 9.50816320f,
    },
    {
        9.51365692f, 9.51915381f, 9.52465388f, 9.53015712f, 9.53566355f,
        9.54117315f, 9.54668594f, 9.55220192f, 9.55772108f, 9.56324343f,
        9.56876897f, 9.57429771f, 9.57982964f, 9.58536476f, 9.59090309f,
        9.59644461f, 9.60198933f, 9.60753726f, 9.61308840f, 9.61864274f,
        9.62420029f, 9.62976105f, 9.63532503f, 9.64089222f, 9.64646262f,
        9.65203625f, 9.65761309f, 9.66319316f, 9.66877645f, 9.67436297f,
        9.67995271f, 9.68554569f, 9.69114190f, 9.69674134f, 9.70234401f,
        9.70794992f, 9.71355908f, 9.71917147f, 9.72478710f, 9.73040598f,
        9.73602811f, 9.74165348f, 9.74728211f, 9.75291399f, 9.75854912f,
        9.76418751f, 9.76982915f, 9.77547406f, 9.78112222f, 9.78677365f,
        9.79242835f, 9.79808631f, 9.80374754f, 9.80941204f, 9.81507982f,
        9.82075087f, 9.82642519f, 9.83210280f, 9.83778369f, 9.84346785f,
        9.84915531f, 9.85484605f, 9.86054007f, 9.86623739f, 9.87193800f,
        9.87764190f, 9.88334910f, 9.88905959f, 9.89477339f, 9.90049048f,
        9.90621088f, 9.91193459f, 9.91766160f, 9.92339192f, 9.92912555f,
        9.93486250f, 9.94060276f, 9.94634633f, 9.95209323f, 9.95784344f,
        9.96359698f, 9.96935384f, 9.97511403f, 9.98087754f, 9.98664439f,
        9.99241457f, 9.99818808f, 10.00396493f, 10.00974512f, 10.01552864f,
        10.02131551f, 10.02710572f, 10.03289928f, 10.03869618f, 10.04449643f,
        10.05030004f, 10.05610700f, 10.06191731f, 10.06773098f, 10.07354801f,
    },
    {
        10.07936840f, 10.08519215f, 10.09101927f, 10.09684976f, 10.10268361f,
        10.10852083f, 10.11436143f, 10.12020540f, 10.12605275f, 10.13190348f,
        10.13775759f, 10.14361508f, 10.14947595f, 10.15534021f, 10.16120786f,
        10.16707890f, 10.17295333f, 10.17883116f, 10.18471238f, 10.19059700f,
        10.19648502f, 10.20237644f, 10.20827127f, 10.21416950f, 10.22007114f,
        10.22597619f, 10.23188465f, 10.23779653f, 10.24371182f, 10.24963053f,
        10.25555266f, 10.26147821f, 10.26740718f, 10.27333958f, 10.27927541f,
        10.28521467f, 10.29115736f, 10.29710348f, 10.30305304f, 10.30900603f,
        10.31496247f, 10.32092234f, 10.32688567f, 10.33285243f, 10.33882265f,
        10.34479631f, 10.35077342f, 10.35675399f, 10.36273802f, 10.36872550f,
        10.37471644f, 10.38071084f, 10.38670871f, 10.39271004f, 10.39871484f,
        10.40472310f, 10.41073484f, 10.41675006f, 10.42276875f, 10.42879091f,
        10.43481656f, 10.44084569f, 10.44687830f, 10.45291439f, 10.45895398f,
        10.46499705f, 10.47104362f, 10.47709368f, 10.48314723f, 10.48920428f,
        10.49526484f, 10.50132889f, 10.50739645f, 10.51346751f, 10.51954208f,
        10.52562016f, 10.53170176f, 10.53778686f, 10.54387548f, 10.54996763f,
        10.55606329f, 10.56216247f, 10.56826518f, 10.57437141f, 10.58048117f,
        10.58659446f, 10.59271128f, 10.59883164f, 10.60495553f, 10.61108297f,
        10.61721394f, 10.62334845f, 10.62948651f, 10.63562812f, 10.64177327f,
        10.64792198f, 10.65407424f, 10.66023005f, 10.66638942f, 10.67255234f,
    },
    {
        10.67871883f, 10.68488889f, 10.69106250f, 10.69723969f, 10.70342044f,
        10.70960476f, 10.71579266f, 10.72198413f, 10.72817918f, 10.73437781f,
        10.74058002f, 10.74678582f, 10.75299520f, 10.75920816f, 10.76542472f,
        10.77164487f, 10.77786861f, 10.78409595f, 10.79032689f, 10.79656143f,
        10.80279957f, 10.80904131f, 10.81528666f, 10.82153562f, 10.82778819f,
        10.83404438f, 10.84030417f, 10.84656759f, 10.85283462f, 10.85910527f,
        10.86537955f, 10.87165745f, 10.87793898f, 10.88422414f, 10.89051293f,
        10.89680536f, 10.90310142f, 10.90940111f, 10.91570445f, 10.92201143f,
        10.92832205f, 10.93463632f, 10.94095424f, 10.94727581f, 10.95360103f,
        10.95992991f, 10.96626244f, 10.97259863f, 10.97893848f, 10.98528200f,
        10.99162918f, 10.99798003f, 11.00433455f, 11.01069273f, 11.01705460f,
        11.02342014f, 11.02978935f, 11.03616225f, 11.04253883f, 11.04891909f,
        11.05530304f, 11.06169068f, 11.06808201f, 11.07447703f, 11.08087574f,
        11.08727816f, 11.09368427f, 11.10009409f, 11.10650760f, 11.11292483f,
        11.11934576f, 11.12577040f, 11.13219875f, 11.13863082f, 11.14506661f,
        11.15150611f, 11.15794933f, 11.16439628f, 11.17084695f, 11.17730135f,
        11.18375947f, 11.19022133f, 11.19668692f, 11.20315625f, 11.20962932f,
        11.21610613f, 11.22258667f, 11.22907097f, 11.23555901f, 11.24205079f,
        11.24854633f, 11.25504563f, 11.26154867f, 11.26805548f, 11.27456604f,
        11.28108037f, 11.28759846f, 11.29412031f, 11.30064594f, 11.30717533f,
    },
    {
        11.31370850f, 11.32024544f, 11.32678616f, 11.33333066f, 11.33987894f,
        11.34643100f, 11.35298685f, 11.35954649f, 11.36610991f, 11.37267713f,
        11.37924815f, 11.38582296f, 11.39240156f, 11.39898397f, 11.40557019f,
        11.41216021f, 11.41875403f, 11.42535167f, 11.43195312f, 11.43855838f,
        11.44516746f, 11.45178036f, 11.45839708f, 11.46501762f, 11.47164198f,
        11.47827018f, 11.48490220f, 11.49153806f, 11.49817775f, 11.50482128f,
        11.51146864f, 11.51811985f, 11.52477489f, 11.53143379f, 11.53809653f,
        11.54476312f, 11.55143356f, 11.55810786f, 11.56478601f, 11.57146803f,
        11.57815390f, 11.58484364f, 11.59153724f, 11.59823471f, 11.60493604f,
        11.61164125f, 11.61835034f, 11.62506330f, 11.63178014f, 11.63850086f,
        11.64522546f, 11.65195395f, 11.65868633f, 11.66542260f, 11.67216275f,
        11.67890681f, 11.68565476f, 11.69240661f, 11.69916236f, 11.70592201f,
        11.71268557f, 11.71945303f, 11.72622441f, 11.73299970f, 11.73977890f,
        11.74656203f, 11.75334907f, 11.76014003f, 11.76693491f, 11.77373373f,
        11.78053646f, 11.78734314f, 11.79415374f, 11.80096828f, 11.80778675f,
        11.81460917f, 11.82143552f, 11.82826583f, 11.83510007f, 11.84193827f,
        11.84878042f, 11.85562652f, 11.86247658f, 11.86933059f, 11.87618857f,
        11.88305050f, 11.88991640f, 11.89678627f, 11.90366011f, 11.91053792f,
        11.91741970f, 11.92430547f, 11.93119520f, 11.93808892f, 11.94498663f,
        11.95188831f, 11.95879399f, 11.96570365f, 11.97261731f, 11.97953497f,
    },
    {
        11.98645662f, 11.99338226f, 12.00031191f, 12.00724557f, 12.01418323f,
        12.02112490f, 12.02807058f, 12.03502027f, 12.04197398f, 12.04893171f,
        12.05589345f, 12.06285922f, 12.06982901f, 12.07680283f, 12.08378068f,
        12.09076257f, 12.09774848f, 12.10473843f, 12.11173242f, 12.11873046f,
        12.12573253f, 12.13273865f, 12.13974882f, 12.14676304f, 12.15378131f,
        12.16080364f, 12.16783003f, 12.17486047f, 12.18189498f, 12.18893355f,
        12.19597619f, 12.20302289f, 12.21007367f, 12.21712852f, 12.22418745f,
        12.23125046f, 12.23831755f, 12.24538872f, 12.25246398f, 12.25954332f,
        12.26662676f, 12.27371429f, 12.28080591f, 12.28790163f, 12.29500145f,
        12.30210537f, 12.30921340f, 12.31632553f, 12.32344178f, 12.33056213f,
        12.33768660f, 12.34481519f, 12.35194789f, 12.35908472f, 12.36622567f,
        12.37337074f, 12.38051995f, 12.38767328f, 12.39483075f, 12.40199235f,
        12.40915809f, 12.41632798f, 12.42350200f, 12.43068017f, 12.43786249f,
        12.44504895f, 12.45223957f, 12.45943434f, 12.46663327f, 12.47383636f,
        12.48104362f, 12.48825503f, 12.49547061f, 12.50269037f, 12.50991429f,
        12.51714239f, 12.52437466f, 12.53161111f, 12.53885175f, 12.54609656f,
        12.55334557f, 12.56059876f, 12.56785614f, 12.57511772f, 12.58238349f,
        12.58965346f, 12.59692763f, 12.60420600f, 12.61148858f, 12.61877536f,
        12.62606636f, 12.63336157f, 12.64066099f, 12.64796463f, 12.65527249f,
        12.66258458f, 12.66990088f, 12.67722142f, 12.68454619f, 12.69187518f,
    },
    {
        12.69920842f, 12.70654589f, 12.71388759f, 12.72123354f, 12.72858374f,
        12.73593818f, 12.74329687f, 12.75065982f, 12.75802701f, 12.76539847f,
        12.77277418f, 12.78015416f, 12.78753840f, 12.79492690f, 12.80231968f,
        12.80971672f, 12.81711804f, 12.82452364f, 12.83193351f, 12.83934767f,
        12.84676611f, 12.85418884f, 12.86161585f, 12.86904716f, 12.87648276f,
        12.88392266f, 12.89136685f, 12.89881535f, 12.90626815f, 12.91372525f,
        12.92118667f, 12.92865239f, 12.93612243f, 12.94359679f, 12.95107546f,
        12.95855846f, 12.96604578f, 12.97353742f, 12.98103340f, 12.98853370f,
        12.99603834f, 13.00354732f, 13.01106063f, 13.01857828f, 13.02610028f,
        13.03362663f, 13.04115732f, 13.04869236f, 13.05623176f, 13.06377551f,
        13.07132363f, 13.07887610f, 13.08643294f, 13.09399414f, 13.10155971f,
        13.10912966f, 13.11670397f, 13.12428267f, 13.13186574f, 13.13945320f,
        13.14704503f, 13.15464126f, 13.16224187f, 13.16984688f, 13.17745628f,
        13.18507007f, 13.19268827f, 13.20031086f, 13.20793787f, 13.21556927f,
        13.22320509f, 13.23084532f, 13.23848996f, 13.24613902f, 13.25379250f,
        13.26145041f, 13.26911273f, 13.27677949f, 13.28445067f, 13.29212629f,
        13.29980634f, 13.30749083f, 13.31517976f, 13.32287313f, 13.33057094f,
        13.33827321f, 13.34597992f, 13.35369109f, 13.36140671f, 13.36912679f,
        13.37685133f, 13.38458034f, 13.39231381f, 13.40005175f, 13.40779415f,
        13.41554104f, 13.42329240f, 13.43104823f, 13.43880855f, 13.44657336f,
    },
    {
        13.45434264f, 13.46211642f, 13.46989469f, 13.47767746f, 13.48546472f,
        13.49325648f, 13.50105274f, 13.50885350f, 13.51665878f, 13.52446856f,
        13.53228286f, 13.54010167f, 13.54792500f, 13.55575285f, 13.56358522f,
        13.57142212f, 13.57926354f, 13.58710950f, 13.59495999f, 13.60281501f,
        13.61067458f, 13.61853868f, 13.62640733f, 13.63428052f, 13.64215827f,
        13.65004056f, 13.65792741f, 13.66581882f, 13.67371479f, 13.68161532f,
        13.68952041f, 13.69743007f, 13.70534430f, 13.71326311f, 13.72118649f,
        13.72911444f, 13.73704698f, 13.74498410f, 13.75292581f, 13.76087211f,
        13.76882300f, 13.77677848f, 13.78473856f, 13.79270323f, 13.80067251f,
        13.80864640f, 13.81662489f, 13.82460799f, 13.83259570f, 13.84058803f,
        13.84858498f, 13.85658654f, 13.86459273f, 13.87260355f, 13.88061899f,
        13.88863907f, 13.89666378f, 13.90469313f, 13.91272711f, 13.92076574f,
        13.92880901f, 13.93685693f, 13.94490950f, 13.95296672f, 13.96102860f,
        13.96909514f, 13.97716633f, 13.98524219f, 13.99332272f, 14.00140792f,
        14.00949778f, 14.01759232f, 14.02569154f, 14.03379544f, 14.04190402f,
        14.05001728f, 14.05813524f, 14.06625788f, 14.07438521f, 14.08251725f,
        14.09065398f, 14.09879541f, 14.10694155f, 14.11509239f, 14.12324794f,
        14.13140821f, 14.13957318f, 14.14774288f, 14.15591730f, 14.16409644f,
        14.17228031f, 14.18046890f, 14.18866223f, 14.19686029f, 14.20506308f,
        14.21327062f, 14.22148290f, 14.22969992f, 14.23792169f, 14.24614822f,
    },
    {
        14.25437949f, 14.26261552f, 14.27085631f, 14.27910186f, 14.28735218f,
        14.29560726f, 14.30386711f, 14.31213174f, 14.32040113f, 14.32867531f,
        14.33695427f, 14.34523801f, 14.35352654f, 14.36181986f, 14.37011797f,
        14.37842087f, 14.38672857f, 14.39504107f, 14.40335837f, 14.41168048f,
        14.42000740f, 14.42833913f, 14.43667567f, 14.44501703f, 14.45336321f,
        14.46171421f, 14.47007004f, 14.47843070f, 14.48679618f, 14.49516650f,
        14.50354165f, 14.51192165f, 14.52030649f, 14.52869617f, 14.53709069f,
        14.54549007f, 14.55389430f, 14.56230339f, 14.57071734f, 14.57913615f,
        14.58755982f, 14.59598836f, 14.60442176f, 14.61286005f, 14.62130320f,
        14.62975124f, 14.63820416f, 14.64666196f, 14.65512464f, 14.66359222f,
        14.67206469f, 14.68054206f, 14.68902432f, 14.69751148f, 14.70600355f,
        14.71450053f, 14.72300241f, 14.73150921f, 14.74002092f, 14.74853755f,
        14.75705910f, 14.76558557f, 14.77411697f, 14.78265330f, 14.79119456f,
        14.79974076f, 14.80829190f, 14.81684797f, 14.82540899f, 14.83397496f,
        14.84254587f, 14.85112174f, 14.85970256f, 14.86828834f, 14.87687908f,
        14.88547479f, 14.89407546f, 14.90268110f, 14.91129171f, 14.91990730f,
        14.92852786f, 14.93715341f, 14.94578394f, 14.95441946f, 14.96305997f,
        14.97170546f, 14.98035596f, 14.98901145f, 14.99767194f, 15.00633744f,
        15.01500795f, 15.02368346f, 15.03236399f, 15.04104953f, 15.04974009f,
        15.05843567f, 15.06713628f, 15.07584191f, 15.08455257f, 15.09326827f,
    },
    {
        15.10198900f, 15.11071477f, 15.11944559f, 15.12818144f, 15.13692235f,
        15.14566830f, 15.15441931f, 15.16317538f, 15.17193650f, 15.18070268f,
        15.18947394f, 15.19825025f, 15.20703164f, 15.21581811f, 15.22460965f,
        15.23340627f, 15.24220797f, 15.25101476f, 15.25982663f, 15.26864360f,
        15.27746566f, 15.28629282f, 15.29512508f, 15.30396244f, 15.31280491f,
        15.32165249f, 15.33050518f, 15.33936299f, 15.34822591f, 15.35709395f,
        15.36596712f, 15.37484542f, 15.38372884f, 15.39261740f, 15.40151109f,
        15.41040992f, 15.41931389f, 15.42822301f, 15.43713728f, 15.44605669f,
        15.45498126f, 15.46391099f, 15.47284587f, 15.48178592f, 15.49073114f,
        15.49968152f, 15.50863707f, 15.51759780f, 15.52656370f, 15.53553479f,
        15.54451106f, 15.55349251f, 15.56247916f, 15.57147100f, 15.58046803f,
        15.58947026f, 15.59847769f, 15.60749033f, 15.61650817f, 15.62553123f,
        15.63455949f, 15.64359298f, 15.65263168f, 15.66167561f, 15.67072476f,
        15.67977914f, 15.68883875f, 15.69790360f, 15.70697368f, 15.71604901f,
        15.72512958f, 15.73421539f, 15.74330646f, 15.75240277f, 15.76150435f,
        15.77061118f, 15.77972327f, 15.78884063f, 15.79796326f, 15.80709115f,
        15.81622433f, 15.82536277f, 15.83450650f, 15.84365551f, 15.85280981f,
        15.86196940f, 15.87113428f, 15.88030445f, 15.88947993f, 15.89866070f,
        15.90784678f, 15.91703817f, 15.92623487f, 15.93543688f, 15.94464421f,
        15.95385685f, 15.96307482f, 15.97229812f, 15.98152675f, 15.99076071f,
    },
    {
        16.00000000f, 16.00924463f, 16.01849461f, 16.02774992f, 16.03701059f,
        16.04627661f, 16.05554798f, 16.06482470f, 16.07410679f, 16.08339424f,
        16.09268706f, 16.10198524f, 16.11128880f, 16.12059773f, 16.12991205f,
        16.13923174f, 16.14855682f, 16.15788729f, 16.16722314f, 16.17656440f,
        16.18591104f, 16.19526309f, 16.20462055f, 16.21398341f, 16.22335168f,
        16.23272536f, 16.24210446f, 16.25148897f, 16.26087891f, 16.27027428f,
        16.27967507f, 16.28908130f, 16.29849296f, 16.30791006f, 16.31733260f,
        16.32676058f, 16.33619401f, 16.34563289f, 16.35507723f, 16.36452702f,
        16.37398227f, 16.38344299f, 16.39290917f, 16.40238082f, 16.41185794f,
        16.42134054f, 16.43082862f, 16.44032218f, 16.44982123f, 16.45932576f,
        16.46883579f, 16.47835131f, 16.48787233f, 16.49739884f, 16.50693087f,
        16.51646840f, 16.52601144f, 16.53556000f, 16.54511407f, 16.55467366f,
        16.56423878f, 16.57380942f, 16.58338560f, 16.59296730f, 16.60255455f,
        16.61214733f, 16.62174565f, 16.63134952f, 16.64095894f, 16.65057391f,
        16.66019444f, 16.66982053f, 16.67945217f, 16.68908939f, 16.69873217f,
        16.70838052f, 16.71803445f, 16.72769395f, 16.73735904f, 16.74702971f,
        16.75670597f, 16.76638781f, 16.77607526f, 16.78576830f, 16.79546694f,
        16.80517118f, 16.81488103f, 16.82459650f, 16.83431757f, 16.84404426f,
        16.85377658f, 16.86351451f, 16.87325807f, 16.88300726f, 16.89276209f,
        16.90252255f, 16.91228865f, 16.92206039f, 16.93183778f, 16.94162082f,
    },
    {
        16.95140951f, 16.96120386f, 16.97100386f, 16.98080953f, 16.99062086f,
        17.00043787f, 17.01026054f, 17.02008889f, 17.02992292f, 17.03976263f,
        17.04960803f, 17.05945911f, 17.06931589f, 17.07917836f, 17.08904653f,
        17.09892040f, 17.10879998f, 17.11868526f, 17.12857626f, 17.13847297f,
        17.14837540f, 17.15828355f, 17.16819743f, 17.17811703f, 17.18804237f,
        17.19797344f, 17.20791025f, 17.21785280f, 17.22780109f, 17.23775513f,
        17.24771493f, 17.25768048f, 17.26765178f, 17.27762885f, 17.28761168f,
        17.29760029f, 17.30759466f, 17.31759480f, 17.32760073f, 17.33761243f,
        17.34762992f, 17.35765320f, 17.36768227f, 17.37771714f, 17.38775780f,
        17.39780426f, 17.40785653f, 17.41791461f, 17.42797850f, 17.43804820f,
        17.44812372f, 17.45820507f, 17.46829223f, 17.47838523f, 17.48848406f,
        17.49858872f, 17.50869922f, 17.51881556f, 17.52893775f, 17.53906579f,
        17.54919968f, 17.55933942f, 17.56948502f, 17.57963648f, 17.58979381f,
        17.59995701f, 17.61012608f, 17.62030103f, 17.63048185f, 17.64066856f,
        17.65086115f, 17.66105964f, 17.67126401f, 17.68147428f, 17.69169045f,
        17.70191253f, 17.71214051f, 17.72237439f, 17.73261420f, 17.74285991f,
        17.75311155f, 17.76336911f, 17.77363260f, 17.78390202f, 17.79417737f,
        17.80445866f, 17.81474589f, 17.82503906f, 17.83533819f, 17.84564326f,
        17.85595428f, 17.86627127f, 17.87659421f, 17.88692312f, 17.89725799f,
        17.90759884f, 17.91794566f, 17.92829847f, 17.93865725f, 17.94902202f,
    },
    {
        17.95939277f, 17.96976952f, 17.98015227f, 17.99054101f, 18.00093576f,
        18.01133651f, 18.02174327f, 18.03215604f, 18.04257483f, 18.05299964f,
        18.06343048f, 18.07386734f, 18.08431023f, 18.09475915f, 18.10521412f,
        18.11567512f, 18.12614216f, 18.13661526f, 18.14709441f, 18.15757961f,
        18.16807087f, 18.17856819f, 18.18907157f, 18.19958103f, 18.21009655f,
        18.22061816f, 18.23114584f, 18.24167960f, 18.25221945f, 18.26276539f,
        18.27331743f, 18.28387556f, 18.29443979f, 18.30501013f, 18.31558657f,
        18.32616912f, 18.33675779f, 18.34735258f, 18.35795349f, 18.36856052f,
        18.37917368f, 18.38979297f, 18.40041840f, 18.41104997f, 18.42168768f,
        18.43233154f, 18.44298155f, 18.45363771f, 18.46430003f, 18.47496851f,
        18.48564315f, 18.49632396f, 18.50701094f, 18.51770410f, 18.52840343f,
        18.53910895f, 18.54982065f, 18.56053855f, 18.57126263f, 18.58199291f,
        18.59272939f, 18.60347208f, 18.61422097f, 18.62497607f, 18.63573738f,
        18.64650492f, 18.65727867f, 18.66805865f, 18.67884486f, 18.68963730f,
        18.70043598f, 18.71124089f, 18.72205205f, 18.73286946f, 18.74369311f,
        18.75452302f, 18.76535919f, 18.77620162f, 18.78705031f, 18.79790527f,
        18.80876650f, 18.81963401f, 18.83050780f, 18.84138787f, 18.85227422f,
        18.86316687f, 18.87406581f, 18.88497104f, 18.89588258f, 18.90680043f,
        18.91772458f, 18.92865504f, 18.93959182f, 18.95053491f, 18.96148434f,
        18.97244008f, 18.98340216f, 18.99437057f, 19.00534532f, 19.01632641f,
    },
    {
        19.02731384f, 19.03830762f, 19.04930776f, 19.06031425f, 19.07132710f,
        19.08234631f, 19.09337189f, 19.10440384f, 19.11544216f, 19.12648686f,
        19.13753795f, 19.14859542f, 19.15965927f, 19.17072952f, 19.18180617f,
        19.19288922f, 19.20397867f, 19.21507452f, 19.22617679f, 19.23728548f,
        19.24840058f, 19.25952210f, 19.27065005f, 19.28178443f, 19.29292524f,
        19.30407249f, 19.31522618f, 19.32638632f, 19.33755290f, 19.34872594f,
        19.35990543f, 19.37109138f, 19.38228379f, 19.39348267f, 19.40468802f,
        19.41589985f, 19.42711815f, 19.43834294f, 19.44957421f, 19.46081197f,
        19.47205622f, 19.48330697f, 19.49456422f, 19.50582797f, 19.51709824f,
        19.52837501f, 19.53965830f, 19.55094811f, 19.56224444f, 19.57354730f,
        19.58485669f, 19.59617262f, 19.60749508f, 19.61882409f, 19.63015964f,
        19.64150174f, 19.65285039f, 19.66420560f, 19.67556737f, 19.68693571f,
        19.69831061f, 19.70969209f, 19.72108014f, 19.73247478f, 19.74387599f,
        19.75528380f, 19.76669819f, 19.77811918f, 19.78954677f, 19.80098097f,
        19.81242177f, 19.82386918f, 19.83532320f, 19.84678384f, 19.85825110f,
        19.86972499f, 19.88120551f, 19.89269266f, 19.90418645f, 19.91568688f,
        19.92719396f, 19.93870768f, 19.95022806f, 19.96175509f, 19.97328878f,
        19.98482914f, 19.99637616f, 20.00792986f, 20.01949023f, 20.03105728f,
        20.04263102f, 20.05421144f, 20.06579855f, 20.07739236f, 20.08899287f,
        20.10060008f, 20.11221399f, 20.12383462f, 20.13546196f, 20.14709602f,
    },
    {
        20.15873680f, 20.17038430f, 20.18203854f, 20.19369951f, 20.20536722f,
        20.21704167f, 20.22872286f, 20.24041081f, 20.25210550f, 20.26380696f,
        20.27551517f, 20.28723015f, 20.29895190f, 20.31068042f, 20.32241572f,
        20.33415780f, 20.34590666f, 20.35766231f, 20.36942476f, 20.38119400f,
        20.39297004f, 20.40475288f, 20.41654253f, 20.42833900f, 20.44014228f,
        20.45195238f, 20.46376930f, 20.47559305f, 20.48742363f, 20.49926105f,
        20.51110531f, 20.52295641f, 20.53481436f, 20.54667916f, 20.55855082f,
        20.57042933f, 20.58231471f, 20.59420696f, 20.60610607f, 20.61801206f,
        20.62992494f, 20.64184469f, 20.65377133f, 20.66570486f, 20.67764529f,
        20.68959262f, 20.70154685f, 20.71350798f, 20.72547603f, 20.73745099f,
        20.74943287f, 20.76142168f, 20.77341741f, 20.78542007f, 20.79742967f,
        20.80944621f, 20.82146969f, 20.83350011f, 20.84553749f, 20.85758182f,
        20.86963312f, 20.88169137f, 20.89375659f, 20.90582879f, 20.91790796f,
        20.92999410f, 20.94208723f, 20.95418735f, 20.96629446f, 20.97840857f,
        20.99052967f, 21.00265778f, 21.01479289f, 21.02693502f, 21.03908416f,
        21.05124033f, 21.06340351f, 21.07557372f, 21.08775097f, 21.09993525f,
        21.11212657f, 21.12432494f, 21.13653035f, 21.14874282f, 21.16096234f,
        21.17318892f, 21.18542257f, 21.19766328f, 21.20991107f, 21.22216593f,
        21.23442788f, 21.24669691f, 21.25897303f, 21.27125624f, 21.28354655f,
        21.29584396f, 21.30814847f, 21.32046010f, 21.33277883f, 21.34510469f,
    },
    {
        21.35743767f, 21.36977777f, 21.38212500f, 21.39447937f, 21.40684088f,
        21.41920952f, 21.43158532f, 21.44396826f, 21.45635836f, 21.46875562f,
        21.48116004f, 21.49357163f, 21.50599039f, 21.51841633f, 21.53084944f,
        21.54328974f, 21.55573723f, 21.56819191f, 21.58065378f, 21.59312286f,
        21.60559914f, 21.61808263f, 21.63057333f, 21.64307125f, 21.65557639f,
        21.66808875f, 21.68060835f, 21.69313517f, 21.70566924f, 21.71821055f,
        21.73075910f, 21.74331491f, 21.75587797f, 21.76844828f, 21.78102586f,
        21.79361071f, 21.80620283f, 21.81880223f, 21.83140890f, 21.84402286f,
        21.85664411f, 21.86927265f, 21.88190848f, 21.89455162f, 21.90720206f,
        21.91985981f, 21.93252488f, 21.94519726f, 21.95787697f, 21.97056400f,
        21.98325836f, 21.99596006f, 22.00866909f, 22.02138547f, 22.03410919f,
        22.04684027f, 22.05957870f, 22.07232450f, 22.08507765f, 22.09783818f,
        22.11060608f, 22.12338136f, 22.13616401f, 22.14895406f, 22.16175149f,
        22.17455632f, 22.18736854f, 22.20018817f, 22.21301521f, 22.22584965f,
        22.23869152f, 22.25154080f, 22.26439751f, 22.27726164f, 22.29013321f,
        22.30301222f, 22.31589866f, 22.32879255f, 22.34169390f, 22.35460269f,
        22.36751895f, 22.38044266f, 22.39337385f, 22.40631251f, 22.41925864f,
        22.43221225f, 22.44517335f, 22.45814193f, 22.47111801f, 22.48410159f,
        22.49709267f, 22.51009125f, 22.52309734f, 22.53611095f, 22.54913208f,
        22.56216073f, 22.57519691f, 22.58824063f, 22.60129187f, 22.61435066f,
    },
    {
        22.62741700f, 22.64049088f, 22.65357232f, 22.66666132f, 22.67975788f,
        22.69286200f, 22.70597370f, 22.71909297f, 22.73221983f, 22.74535426f,
        22.75849629f, 22.77164591f, 22.78480313f, 22.79796795f, 22.81114038f,
        22.82432041f, 22.83750807f, 22.85070334f, 22.86390624f, 22.87711676f,
        22.89033492f, 22.90356071f, 22.91679415f, 22.93003523f, 22.94328397f,
        22.95654036f, 22.96980441f, 22.98307612f, 22.99635550f, 23.00964255f,
        23.02293728f, 23.03623969f, 23.04954979f, 23.06286758f, 23.07619306f,
        23.08952624f, 23.10286713f, 23.11621572f, 23.12957203f, 23.14293605f,
        23.15630780f, 23.16968727f, 23.18307447f, 23.19646941f, 23.20987209f,
        23.22328251f, 23.23670068f, 23.25012660f, 23.26356028f, 23.27700172f,
        23.29045093f, 23.30390790f, 23.31737266f, 23.33084519f, 23.34432551f,
        23.35781361f, 23.37130951f, 23.38481321f, 23.39832471f, 23.41184402f,
        23.42537114f, 23.43890607f, 23.45244882f, 23.46599940f, 23.47955781f,
        23.49312405f, 23.50669813f, 23.52028006f, 23.53386983f, 23.54746745f,
        23.56107293f, 23.57468627f, 23.58830748f, 23.60193655f, 23.61557350f,
        23.62921834f, 23.64287105f, 23.65653165f, 23.67020015f, 23.68387654f,
        23.69756084f, 23.71125304f, 23.72495315f, 23.73866118f, 23.75237713f,
        23.76610100f, 23.77983281f, 23.79357255f, 23.80732022f, 23.82107584f,
        23.83483941f, 23.84861093f, 23.86239041f, 23.87617785f, 23.88997325f,
        23.90377663f, 23.91758798f, 23.93140731f, 23.94523463f, 23.95906993f,
    },
    {
        23.97291323f, 23.98676453f, 24.00062383f, 24.01449114f, 24.02836646f,
        24.04224980f, 24.05614116f, 24.07004054f, 24.08394796f, 24.09786341f,
        24.11178690f, 24.12571844f, 24.13965803f, 24.15360567f, 24.16756137f,
        24.18152513f, 24.19549696f, 24.20947687f, 24.22346485f, 24.23746091f,
        24.25146506f, 24.26547731f, 24.27949764f, 24.29352608f, 24.30756263f,
        24.32160728f, 24.33566005f, 24.34972094f, 24.36378995f, 24.37786710f,
        24.39195237f, 24.40604579f, 24.42014734f, 24.43425705f, 24.44837491f,
        24.46250092f, 24.47663510f, 24.49077744f, 24.50492795f, 24.51908664f,
        24.53325351f, 24.54742857f, 24.56161182f, 24.57580326f, 24.59000290f,
        24.60421075f, 24.61842680f, 24.63265107f, 24.64688356f, 24.66112427f,
        24.67537321f, 24.68963038f, 24.70389579f, 24.71816944f, 24.73245134f,
        24.74674149f, 24.76103990f, 24.77534657f, 24.78966150f, 24.80398471f,
        24.81831619f, 24.83265595f, 24.84700400f, 24.86136034f, 24.87572497f,
        24.89009790f, 24.90447914f, 24.91886869f, 24.93326655f, 24.94767273f,
        24.96208723f, 24.97651006f, 24.99094123f, 25.00538073f, 25.01982858f,
        25.03428477f, 25.04874932f, 25.06322223f, 25.07770349f, 25.09219313f,
        25.10669113f, 25.12119752f, 25.13571228f, 25.15023543f, 25.16476697f,
        25.17930691f, 25.19385525f, 25.20841200f, 25.22297715f, 25.23755072f,
        25.25213271f, 25.26672313f, 25.28132198f, 25.29592926f, 25.31054498f,
        25.32516915f, 25.33980177f, 25.35444284f, 25.36909237f, 25.38375037f,
    },
    {
        25.39841683f, 25.41309177f, 25.42777519f, 25.44246709f, 25.45716748f,
        25.47187636f, 25.48659375f, 25.50131963f, 25.51605403f, 25.53079694f,
        25.54554836f, 25.56030831f, 25.57507679f, 25.58985380f, 25.60463935f,
        25.61943344f, 25.63423608f, 25.64904728f, 25.66386703f, 25.67869534f,
        25.69353222f, 25.70837767f, 25.72323170f, 25.73809432f, 25.75296552f,
        25.76784531f, 25.78273370f, 25.79763069f, 25.81253630f, 25.82745051f,
        25.84237334f, 25.85730479f, 25.87224487f, 25.88719358f, 25.90215093f,
        25.91711692f, 25.93209156f, 25.94707485f, 25.96206680f, 25.97706741f,
        25.99207668f, 26.00709463f, 26.02212126f, 26.03715657f, 26.05220056f,
        26.06725325f, 26.08231464f, 26.09738472f, 26.11246352f, 26.12755103f,
        26.14264725f, 26.15775220f, 26.17286587f, 26.18798828f, 26.20311943f,
        26.21825931f, 26.23340795f, 26.24856534f, 26.26373148f, 26.27890639f,
        26.29409007f, 26.30928252f, 26.32448374f, 26.33969375f, 26.35491255f,
        26.37014014f, 26.38537653f, 26.40062173f, 26.41587573f, 26.43113855f,
        26.44641018f, 26.46169064f, 26.47697993f, 26.49227805f, 26.50758501f,
        26.52290081f, 26.53822547f, 26.55355897f, 26.56890134f, 26.58425257f,
        26.59961268f, 26.61498165f, 26.63035951f, 26.64574625f, 26.66114189f,
        26.67654641f, 26.69195984f, 26.70738218f, 26.72281342f, 26.73825358f,
        26.75370267f, 26.76916067f, 26.78462761f, 26.80010349f, 26.81558831f,
        26.83108207f, 26.84658479f, 26.86209647f, 26.87761711f, 26.89314671f,
    },
    {
        26.90868529f, 26.92423284f, 26.93978938f, 26.95535491f, 26.97092943f,
        26.98651295f, 27.00210548f, 27.01770701f, 27.03331756f, 27.04893712f,
        27.06456572f, 27.08020334f, 27.09585000f, 27.11150569f, 27.12717044f,
        27.14284423f, 27.15852708f, 27.17421899f, 27.18991997f, 27.20563002f,
        27.22134915f, 27.23707736f, 27.25281466f, 27.26856105f, 27.28431654f,
        27.30008113f, 27.31585483f, 27.33163764f, 27.34742958f, 27.36323063f,
        27.37904082f, 27.39486014f, 27.41068861f, 27.42652622f, 27.44237297f,
        27.45822889f, 27.47409397f, 27.48996821f, 27.50585162f, 27.52174422f,
        27.53764599f, 27.55355695f, 27.56947711f, 27.58540647f, 27.60134502f,
        27.61729279f, 27.63324977f, 27.64921597f, 27.66519140f, 27.68117606f,
        27.69716995f, 27.71317309f, 27.72918547f, 27.74520710f, 27.76123799f,
        27.77727814f, 27.79332756f, 27.80938625f, 27.82545423f, 27.84153148f,
        27.85761803f, 27.87371386f, 27.88981900f, 27.90593345f, 27.92205720f,
        27.93819028f, 27.95433267f, 27.97048439f, 27.98664544f, 28.00281583f,
        28.01899556f, 28.03518465f, 28.05138308f, 28.06759088f, 28.08380804f,
        28.10003457f, 28.11627047f, 28.13251576f, 28.14877043f, 28.16503449f,
        28.18130796f, 28.19759082f, 28.21388309f, 28.23018478f, 28.24649588f,
        28.26281641f, 28.27914637f, 28.29548576f, 28.31183460f, 28.32819288f,
        28.34456061f, 28.36093780f, 28.37732445f, 28.39372057f, 28.41012617f,
        28.42654124f, 28.44296580f, 28.45939984f, 28.47584339f, 28.49229643f,
    },
    {
        28.50875898f, 28.52523104f, 28.54171262f, 28.55820372f, 28.57470435f,
        28.59121452f, 28.60773422f, 28.62426347f, 28.64080227f, 28.65735062f,
        28.67390854f, 28.69047602f, 28.70705308f, 28.72363971f, 28.74023593f,
        28.75684174f, 28.77345714f, 28.79008214f, 28.80671675f, 28.82336097f,
        28.84001480f, 28.85667826f, 28.87335135f, 28.89003407f, 28.90672643f,
        28.92342843f, 28.94014008f, 28.95686139f, 28.97359236f, 28.99033300f,
        29.00708331f, 29.02384330f, 29.04061297f, 29.05739233f, 29.07418139f,
        29.09098014f, 29.10778861f, 29.12460678f, 29.14143467f, 29.15827229f,
        29.17511963f, 29.19197671f, 29.20884353f, 29.22572009f, 29.24260641f,
        29.25950248f, 29.27640831f, 29.29332391f, 29.31024929f, 29.32718444f,
        29.34412938f, 29.36108411f, 29.37804864f, 29.39502297f, 29.41200710f,
        29.42900105f, 29.44600482f, 29.46301841f, 29.48004184f, 29.49707509f,
        29.51411819f, 29.53117114f, 29.54823394f, 29.56530660f, 29.58238913f,
        29.59948152f, 29.61658379f, 29.63369594f, 29.65081798f, 29.66794991f,
        29.68509174f, 29.70224348f, 29.71940512f, 29.73657668f, 29.75375816f,
        29.77094957f, 29.78815092f, 29.80536220f, 29.82258342f, 29.83981460f,
        29.85705573f, 29.87430682f, 29.89156788f, 29.90883892f, 29.92611993f,
        29.94341093f, 29.96071192f, 29.97802290f, 29.99534389f, 30.01267488f,
        30.03001589f, 30.04736692f, 30.06472797f, 30.08209906f, 30.09948018f,
        30.11687134f, 30.13427256f, 30.15168382f, 30.16910515f, 30.18653654f,
    },
    {
        30.20397801f, 30.22142955f, 30.23889117f, 30.25636289f, 30.27384470f,
        30.29133661f, 30.30883862f, 30.32635075f, 30.34387300f, 30.36140537f,
        30.37894787f, 30.39650051f, 30.41406329f, 30.43163621f, 30.44921929f,
        30.46681253f, 30.48441594f, 30.50202951f, 30.51965327f, 30.53728720f,
        30.55493133f, 30.57258564f, 30.59025016f, 30.60792489f, 30.62560983f,
        30.64330498f, 30.66101036f, 30.67872597f, 30.69645182f, 30.71418791f,
        30.73193424f, 30.74969083f, 30.76745768f, 30.78523479f, 30.80302218f,
        30.82081984f, 30.83862779f, 30.85644602f, 30.87427455f, 30.89211339f,
        30.90996253f, 30.92782198f, 30.94569175f, 30.96357184f, 30.98146227f,
        30.99936304f, 31.01727414f, 31.03519560f, 31.05312741f, 31.07106958f,
        31.08902212f, 31.10698503f, 31.12495832f, 31.14294199f, 31.16093606f,
        31.17894052f, 31.19695538f, 31.21498066f, 31.23301634f, 31.25106245f,
        31.26911899f, 31.28718596f, 31.30526337f, 31.32335122f, 31.34144952f,
        31.35955828f, 31.37767751f, 31.39580720f, 31.41394737f, 31.43209802f,
        31.45025915f, 31.46843078f, 31.48661291f, 31.50480555f, 31.52300869f,
        31.54122236f, 31.55944654f, 31.57768126f, 31.59592651f, 31.61418231f,
        31.63244865f, 31.65072555f, 31.66901301f, 31.68731103f, 31.70561962f,
        31.72393880f, 31.74226856f, 31.76060891f, 31.77895985f, 31.79732140f,
        31.81569356f, 31.83407634f, 31.85246973f, 31.87087375f, 31.88928841f,
        31.90771371f, 31.92614965f, 31.94459624f, 31.96305350f, 31.98152141f,
    },
    {
        32.00000000f, 32.01848926f, 32.03698921f, 32.05549985f, 32.07402118f,
        32.09255321f, 32.11109595f, 32.12964941f, 32.14821358f, 32.16678848f,
        32.18537411f, 32.20397049f, 32.22257760f, 32.24119547f, 32.25982409f,
        32.27846348f, 32.29711364f, 32.31577457f, 32.33444629f, 32.35312879f,
        32.37182209f, 32.39052619f, 32.40924109f, 32.42796681f, 32.44670335f,
        32.46545072f, 32.48420891f, 32.50297795f, 32.52175783f, 32.54054856f,
        32.55935015f, 32.57816260f, 32.59698592f, 32.61582012f, 32.63466519f,
        32.65352116f, 32.67238802f, 32.69126579f, 32.71015446f, 32.72905404f,
        32.74796454f, 32.76688597f, 32.78581834f, 32.80476164f, 32.82371589f,
        32.84268108f, 32.86165724f, 32.88064436f, 32.89964245f, 32.91865152f,
        32.93767157f, 32.95670261f, 32.97574465f, 32.99479769f, 33.01386174f,
        33.03293680f, 33.05202288f, 33.07112000f, 33.09022814f, 33.10934733f,
        33.12847756f, 33.14761885f, 33.16677120f, 33.18593461f, 33.20510909f,
        33.22429466f, 33.24349131f, 33.26269905f, 33.28191788f, 33.30114783f,
        33.32038888f, 33.33964105f, 33.35890435f, 33.37817877f, 33.39746433f,
        33.41676104f, 33.43606889f, 33.45538790f, 33.47471807f, 33.49405941f,
        33.51341193f, 33.53277563f, 33.55215051f, 33.57153659f, 33.59093388f,
        33.61034237f, 33.62976207f, 33.64919299f, 33.66863514f, 33.68808853f,
        33.70755315f, 33.72702902f, 33.74651614f, 33.76601453f, 33.78552418f,
        33.80504510f, 33.82457730f, 33.84412078f, 33.86367556f, 33.88324164f,
    },
    {
        33.90281902f, 33.92240771f, 33.94200772f, 33.96161906f, 33.98124173f,
        34.00087573f, 34.02052108f, 34.04017778f, 34.05984584f, 34.07952526f,
        34.09921605f, 34.11891822f, 34.13863177f, 34.15835672f, 34.17809306f,
        34.19784080f, 34.21759995f, 34.23737052f, 34.25715252f, 34.27694594f,
        34.29675080f, 34.31656710f, 34.33639486f, 34.35623406f, 34.37608474f,
        34.39594688f, 34.41582049f, 34.43570559f, 34.45560218f, 34.47551027f,
        34.49542986f, 34.51536095f, 34.53530357f, 34.55525770f, 34.57522337f,
        34.59520057f, 34.61518931f, 34.63518961f, 34.65520146f, 34.67522487f,
        34.69525985f, 34.71530641f, 34.73536455f, 34.75543428f, 34.77551560f,
        34.79560853f, 34.81571307f, 34.83582922f, 34.85595700f, 34.87609640f,
        34.89624745f, 34.91641013f, 34.93658447f, 34.95677046f, 34.97696811f,
        34.99717744f, 35.01739844f, 35.03763113f, 35.05787550f, 35.07813157f,
        35.09839935f, 35.11867884f, 35.13897004f, 35.15927297f, 35.17958763f,
        35.19991402f, 35.22025217f, 35.24060206f, 35.26096371f, 35.28133712f,
        35.30172231f, 35.32211927f, 35.34252802f, 35.36294857f, 35.38338091f,
        35.40382505f, 35.42428101f, 35.44474879f, 35.46522839f, 35.48571983f,
        35.50622311f, 35.52673823f, 35.54726521f, 35.56780404f, 35.58835475f,
        35.60891732f, 35.62949178f, 35.65007813f, 35.67067637f, 35.69128651f,
        35.71190856f, 35.73254253f, 35.75318842f, 35.77384624f, 35.79451599f,
        35.81519768f, 35.83589133f, 35.85659693f, 35.87731450f, 35.89804403f,
    },
    {
        35.91878555f, 35.93953904f, 35.96030453f, 35.98108202f, 36.00187151f,
        36.02267301f, 36.04348654f, 36.06431208f, 36.08514967f, 36.10599929f,
        36.12686095f, 36.14773467f, 36.16862046f, 36.18951831f, 36.21042823f,
        36.23135024f, 36.25228433f, 36.27323052f, 36.29418881f, 36.31515921f,
        36.33614173f, 36.35713637f, 36.37814314f, 36.39916205f, 36.42019311f,
        36.44123631f, 36.46229168f, 36.48335921f, 36.50443891f, 36.52553079f,
        36.54663486f, 36.56775112f, 36.58887958f, 36.61002025f, 36.63117314f,
        36.65233825f, 36.67351558f, 36.69470516f, 36.71590697f, 36.73712104f,
        36.75834736f, 36.77958595f, 36.80083681f, 36.82209994f, 36.84337537f,
        36.86466308f, 36.88596310f, 36.90727542f, 36.92860006f, 36.94993701f,
        36.97128630f, 36.99264792f, 37.01402188f, 37.03540820f, 37.05680687f,
        37.07821790f, 37.09964131f, 37.12107709f, 37.14252526f, 37.16398582f,
        37.18545878f, 37.20694415f, 37.22844193f, 37.24995214f, 37.27147477f,
        37.29300983f, 37.31455734f, 37.33611730f, 37.35768972f, 37.37927460f,
        37.40087195f, 37.42248179f, 37.44410410f, 37.46573891f, 37.48738622f,
        37.50904604f, 37.53071838f, 37.55240323f, 37.57410062f, 37.59581054f,
        37.61753300f, 37.63926802f, 37.66101559f, 37.68277573f, 37.70454844f,
        37.72633374f, 37.74813162f, 37.76994209f, 37.79176517f, 37.81360085f,
        37.83544915f, 37.85731008f, 37.87918363f, 37.90106983f, 37.92296867f,
        37.94488016f, 37.96680432f, 37.98874114f, 38.01069063f, 38.03265281f,
    },
    {
        38.05462768f, 38.07661524f, 38.09861551f, 38.12062849f, 38.14265419f,
        38.16469262f, 38.18674378f, 38.20880768f, 38.23088432f, 38.25297373f,
        38.27507590f, 38.29719083f, 38.31931855f, 38.34145905f, 38.36361234f,
        38.38577843f, 38.40795733f, 38.43014905f, 38.45235358f, 38.47457095f,
        38.49680115f, 38.51904420f, 38.54130010f, 38.56356886f, 38.58585049f,
        38.60814499f, 38.63045237f, 38.65277264f, 38.67510580f, 38.69745187f,
        38.71981086f, 38.74218276f, 38.76456758f, 38.78696534f, 38.80937604f,
        38.83179969f, 38.85423630f, 38.87668587f, 38.89914841f, 38.92162393f,
        38.94411244f, 38.96661394f, 38.98912844f, 39.01165595f, 39.03419647f,
        39.05675002f, 39.07931660f, 39.10189622f, 39.12448889f, 39.14709461f,
        39.16971339f, 39.19234524f, 39.21499016f, 39.23764817f, 39.26031927f,
        39.28300347f, 39.30570078f, 39.32841120f, 39.35113474f, 39.37387142f,
        39.39662123f, 39.41938418f, 39.44216029f, 39.46494955f, 39.48775199f,
        39.51056760f, 39.53339639f, 39.55623837f, 39.57909355f, 39.60196193f,
        39.62484353f, 39.64773835f, 39.67064640f, 39.69356768f, 39.71650221f,
        39.73944999f, 39.76241102f, 39.78538533f, 39.80837291f, 39.83137377f,
        39.85438792f, 39.87741536f, 39.90045611f, 39.92351018f, 39.94657756f,
        39.96965828f, 39.99275233f, 40.01585972f, 40.03898046f, 40.06211457f,
        40.08526204f, 40.10842288f, 40.13159711f, 40.15478472f, 40.17798573f,
        40.20120015f, 40.22442798f, 40.24766924f, 40.27092392f, 40.29419204f,
    },
    {
        40.31747360f, 40.34076861f, 40.36407708f, 40.38739902f, 40.41073444f,
        40.43408334f, 40.45744573f, 40.48082161f, 40.50421101f, 40.52761392f,
        40.55103035f, 40.57446030f, 40.59790380f, 40.62136085f, 40.64483144f,
        40.66831560f, 40.69181332f, 40.71532463f, 40.73884951f, 40.76238799f,
        40.78594007f, 40.80950576f, 40.83308507f, 40.85667799f, 40.88028455f,
        40.90390475f, 40.92753860f, 40.95118610f, 40.97484727f, 40.99852211f,
        41.02221062f, 41.04591282f, 41.06962872f, 41.09335832f, 41.11710163f,
        41.14085866f, 41.16462942f, 41.18841391f, 41.21221214f, 41.23602413f,
        41.25984987f, 41.28368938f, 41.30754266f, 41.33140973f, 41.35529058f,
        41.37918523f, 41.40309369f, 41.42701597f, 41.45095206f, 41.47490199f,
        41.49886575f, 41.52284336f, 41.54683482f, 41.57084015f, 41.59485934f,
        41.61889241f, 41.64293937f, 41.66700023f, 41.69107498f, 41.71516365f,
        41.73926623f, 41.76338274f, 41.78751319f, 41.81165757f, 41.83581591f,
        41.85998821f, 41.88417447f, 41.90837470f, 41.93258892f, 41.95681713f,
        41.98105934f, 42.00531556f, 42.02958579f, 42.05387004f, 42.07816833f,
        42.10248065f, 42.12680702f, 42.15114745f, 42.17550194f, 42.19987050f,
        42.22425314f, 42.24864988f, 42.27306070f, 42.29748563f, 42.32192468f,
        42.34637784f, 42.37084513f, 42.39532656f, 42.41982214f, 42.44433187f,
        42.46885575f, 42.49339381f, 42.51794605f, 42.54251247f, 42.56709309f,
        42.59168791f, 42.61629694f, 42.64092019f, 42.66555767f, 42.69020938f,
    },
    {
        42.71487533f, 42.73955554f, 42.76425001f, 42.78895874f, 42.81368175f,
        42.83841905f, 42.86317064f, 42.88793653f, 42.91271673f, 42.93751125f,
        42.96232009f, 42.98714327f, 43.01198079f, 43.03683266f, 43.06169889f,
        43.08657949f, 43.11147446f, 43.13638382f, 43.16130757f, 43.18624572f,
        43.21119828f, 43.23616525f, 43.26114666f, 43.28614249f, 43.31115277f,
        43.33617750f, 43.36121669f, 43.38627035f, 43.41133848f, 43.43642109f,
        43.46151820f, 43.48662981f, 43.51175593f, 43.53689657f, 43.56205173f,
        43.58722142f, 43.61240566f, 43.63760445f, 43.66281780f, 43.68804572f,
        43.71328822f, 43.73854530f, 43.76381697f, 43.78910324f, 43.81440413f,
        43.83971963f, 43.86504976f, 43.89039452f, 43.91575393f, 43.94112800f,
        43.96651672f, 43.99192011f, 44.01733818f, 44.04277094f, 44.06821839f,
        44.09368054f, 44.11915741f, 44.14464899f, 44.17015531f, 44.19567636f,
        44.22121216f, 44.24676271f, 44.27232803f, 44.29790811f, 44.32350298f,
        44.34911263f, 44.37473709f, 44.40037634f, 44.42603042f, 44.45169931f,
        44.47738303f, 44.50308160f, 44.52879501f, 44.55452328f, 44.58026642f,
        44.60602443f, 44.63179732f, 44.65758511f, 44.68338779f, 44.70920538f,
        44.73503789f, 44.76088533f, 44.78674770f, 44.81262501f, 44.83851728f,
        44.86442450f, 44.89034669f, 44.91628387f, 44.94223602f, 44.96820318f,
        44.99418533f, 45.02018250f, 45.04619469f, 45.07222191f, 45.09826416f,
        45.12432147f, 45.15039383f, 45.17648125f, 45.20258375f, 45.22870133f,
    },
    {
        45.25483400f, 45.28098176f, 45.30714464f, 45.33332263f, 45.35951575f,
        45.38572400f, 45.41194740f, 45.43818595f, 45.46443965f, 45.49070853f,
        45.51699258f, 45.54329182f, 45.56960626f, 45.59593590f, 45.62228075f,
        45.64864083f, 45.67501613f, 45.70140668f, 45.72781247f, 45.75423352f,
        45.78066984f, 45.80712143f, 45.83358830f, 45.86007047f, 45.88656794f,
        45.91308071f, 45.93960881f, 45.96615223f, 45.99271099f, 46.01928510f,
        46.04587456f, 46.07247938f, 46.09909958f, 46.12573515f, 46.15238612f,
        46.17905248f, 46.20573426f, 46.23243145f, 46.25914406f, 46.28587211f,
        46.31261560f, 46.33937454f, 46.36614895f, 46.39293882f, 46.41974417f,
        46.44656501f, 46.47340135f, 46.50025320f, 46.52712055f, 46.55400344f,
        46.58090185f, 46.60781581f, 46.63474531f, 46.66169038f, 46.68865102f,
        46.71562723f, 46.74261903f, 46.76962642f, 46.79664942f, 46.82368803f,
        46.85074227f, 46.87781214f, 46.90489765f, 46.93199880f, 46.95911562f,
        46.98624810f, 47.01339626f, 47.04056011f, 47.06773965f, 47.09493490f,
        47.12214586f, 47.14937254f, 47.17661495f, 47.20387311f, 47.23114701f,
        47.25843667f, 47.28574210f, 47.31306330f, 47.34040030f, 47.36775308f,
        47.39512167f, 47.42250608f, 47.44990630f, 47.47732236f, 47.50475426f,
        47.53220201f, 47.55966562f, 47.58714509f, 47.61464045f, 47.64215168f,
        47.66967882f, 47.69722186f, 47.72478081f, 47.75235569f, 47.77994650f,
        47.80755325f, 47.83517596f, 47.86281462f, 47.89046925f, 47.91813986f,
    },
    {
        47.94582646f, 47.97352906f, 48.00124766f, 48.02898227f, 48.05673292f,
        48.08449959f, 48.11228231f, 48.14008108f, 48.16789592f, 48.19572682f,
        48.22357381f, 48.25143688f, 48.27931606f, 48.30721134f, 48.33512274f,
        48.36305026f, 48.39099393f, 48.41895374f, 48.44692970f, 48.47492183f,
        48.50293013f, 48.53095461f, 48.55899529f, 48.58705217f, 48.61512526f,
        48.64321456f, 48.67132010f, 48.69944188f, 48.72757991f, 48.75573419f,
        48.78390474f, 48.81209157f, 48.84029469f, 48.86851410f, 48.89674981f,
        48.92500184f, 48.95327019f, 48.98155488f, 49.00985591f, 49.03817329f,
        49.06650703f, 49.09485714f, 49.12322364f, 49.15160652f, 49.18000580f,
        49.20842149f, 49.23685360f, 49.26530214f, 49.29376711f, 49.32224854f,
        49.35074641f, 49.37926076f, 49.40779158f, 49.43633888f, 49.46490268f,
        49.49348298f, 49.52207979f, 49.55069313f, 49.57932300f, 49.60796941f,
        49.63663238f, 49.66531190f, 49.69400800f, 49.72272068f, 49.75144994f,
        49.78019581f, 49.80895828f, 49.83773738f, 49.86653310f, 49.89534546f,
        49.92417446f, 49.95302013f, 49.98188246f, 50.01076147f, 50.03965716f,
        50.06856955f, 50.09749864f, 50.12644445f, 50.15540699f, 50.18438625f,
        50.21338227f, 50.24239503f, 50.27142456f, 50.30047086f, 50.32953395f,
        50.35861382f, 50.38771050f, 50.41682399f, 50.44595430f, 50.47510145f,
        50.50426543f, 50.53344626f, 50.56264396f, 50.59185852f, 50.62108997f,
        50.65033830f, 50.67960354f, 50.70888568f, 50.73818474f, 50.76750073f,
    },
    {
        50.79683366f, 50.82618354f, 50.85555038f, 50.88493418f, 50.91433496f,
        50.94375273f, 50.97318749f, 51.00263927f, 51.03210806f, 51.06159387f,
        51.09109673f, 51.12061663f, 51.15015358f, 51.17970760f, 51.20927870f,
        51.23886689f, 51.26847217f, 51.29809455f, 51.32773405f, 51.35739068f,
        51.38706444f, 51.41675535f, 51.44646341f, 51.47618863f, 51.50593104f,
        51.53569062f, 51.56546740f, 51.59526139f, 51.62507259f, 51.65490102f,
        51.68474668f, 51.71460958f, 51.74448974f, 51.77438716f, 51.80430186f,
        51.83423384f, 51.86418312f, 51.89414970f, 51.92413359f, 51.95413481f,
        51.98415337f, 52.01418927f, 52.04424252f, 52.07431314f, 52.10440113f,
        52.13450650f, 52.16462927f, 52.19476945f, 52.22492704f, 52.25510205f,
        52.28529450f, 52.31550440f, 52.34573175f, 52.37597656f, 52.40623885f,
        52.43651863f, 52.46681590f, 52.49713067f, 52.52746296f, 52.55781278f,
        52.58818013f, 52.61856503f, 52.64896749f, 52.67938751f, 52.70982511f,
        52.74028029f, 52.77075307f, 52.80124346f, 52.83175146f, 52.86227709f,
        52.89282036f, 52.92338128f, 52.95395985f, 52.98455610f, 53.01517002f,
        53.04580162f, 53.07645093f, 53.10711795f, 53.13780268f, 53.16850515f,
        53.19922535f, 53.22996331f, 53.26071902f, 53.29149251f, 53.32228377f,
        53.35309283f, 53.38391968f, 53.41476435f, 53.44562684f, 53.47650717f,
        53.50740533f, 53.53832135f, 53.56925523f, 53.60020698f, 53.63117662f,
        53.66216415f, 53.69316959f, 53.72419294f, 53.75523421f, 53.78629342f,
    },
};

void
env_on(Envelope *env) {
    env->stage = ENV_ATTACK;
    env->vol = 0.0f;
    if (env->a > 0) {
        env->a = (SOUND_TIMER / 256.0f) / env->a;
    } else if (env->stage == ENV_ATTACK) {
        env->stage = ENV_DECAY;
        env->vol = 1.0f;
    }
    if (env->d > 0) {
        env->d = (SOUND_TIMER / 256.0f) / env->d;
    } else if (env->stage == ENV_DECAY) {
        env->stage = ENV_SUSTAIN;
        env->vol = env->s;
    }
    if (env->r > 0) {
        env->r = (SOUND_TIMER / 256.0f) / env->r;
    }
}

void
env_off(Envelope *env) {
    env->stage = ENV_RELEASE;
}

void
note_on(AudioChannel *channel, Uint16 duration, Uint8 *data, Uint16 len, Uint8 vol,
        Uint8 attack, Uint8 decay, Uint8 sustain, Uint8 release, Uint8 pitch, bool loop) {
    channel->duration = duration;
    channel->vol_l = (vol >> 4) / 16.0f;
    channel->vol_r = (vol & 0xf) / 16.0f;

    Sample sample = {0};
    sample.data = data;
    sample.len = len;
    sample.pos = 0;
    sample.env.a = attack * 10.0f;
    sample.env.d = decay * 10.0f;
    sample.env.s = sustain / 256.0f;
    sample.env.r = release * 10.0f;
    if (loop) {
        sample.loop = len;
    } else {
        sample.loop = 0;
    }
    env_on(&sample.env);
    const float *inc = &tuning[(pitch + 1) - 21][0];
    sample.inc = *(inc);

    channel->next_sample = sample;
    channel->xfade = true;
}

void
note_off(AudioChannel *channel, Uint16 duration) {
    channel->duration = duration;
    env_off(&channel->sample.env);
}

void
env_advance(Envelope *env) {
    switch (env->stage) {
        case ENV_ATTACK: {
            env->vol += env->a;
            if (env->vol >= 1.0f) {
                env->stage = ENV_DECAY;
                env->vol = 1.0f;
            }
        } break;
        case ENV_DECAY: {
            env->vol -= env->d;
            if (env->vol <= env->s || env->d <= 0) {
                env->stage = ENV_SUSTAIN;
                env->vol = env->s;
            }
        } break;
        case ENV_SUSTAIN: {
            env->vol = env->s;
        } break;
        case ENV_RELEASE: {
            if (env->vol <= 0 || env->r <= 0) {
                env->vol = 0;
            } else {
                env->vol -= env->r;
            }
        } break;
    }
}

float
interpolate_sample(Uint8 *data, Uint16 len, float pos) {
#if INTERPOL_METHOD == 0
    return data[(int)pos];

#elif INTERPOL_METHOD == 1
    float x = pos;
    int x0 = (int)x;
    int x1 = (x0 + 1);
    float y0 = data[x0];
    float y1 = data[x1 % len];
    x = x - x0;
    float y = y0 + x * (y1 - y0);
    return y;

#elif INTERPOL_METHOD == 2
    float x = pos;
    int x0 = x - 1;
    int x1 = x;
    int x2 = x + 1;
    int x3 = x + 2;
    float y0 = data[x0 % len];
    float y1 = data[x1];
    float y2 = data[x2 % len];
    float y3 = data[x3 % len];
    x = x - x1;
    float c0 = y1;
    float c1 = 0.5f * (y2 - y0);
    float c2 = y0 - 2.5f * y1 + 2.f * y2 - 0.5f * y3;
    float c3 = 1.5f * (y1 - y2) + 0.5f * (y3 - y0);
    return ((c3 * x + c2) * x + c1) * x + c0;
#endif
}

Sint16
next_sample(Sample *sample) {
    if (sample->pos >= sample->len) {
        if (sample->loop == 0) {
            sample->data = 0;
            return 0;
        }
        while (sample->pos >= sample->len) {
            sample->pos -= sample->loop;
        }
    }

    float val = interpolate_sample(sample->data, sample->len, sample->pos);
    val *= sample->env.vol;
    Sint8 next = (Sint8)0x80 ^ (Uint8)val;

    sample->pos += sample->inc;
    env_advance(&sample->env);
    return next;
}

int
audio_handler(void *ctx, Uint8 *out_stream, int len) {
    Sint16 *stream = (Sint16 *)out_stream;
    memset(stream, 0x00, len);

    for (int n = 0; n < N_CHANNELS; n++) {
        Uint8 device = (3 + n) << 4;
        // TODO: Make sure this works properly and evals to the audio stack
        // instead of the regular stack.
        // Uxn *u = (Uxn *)ctx;
        // Uint8 *addr = &u->dev[device];
        // if (channel[n].duration <= 0 && PEEK2(addr)) {
			// uxn_eval(u, PEEK2(addr));
        // }
        // printf("DEVICE: %x\n", device);
        // printf("ADDR: %x\n", PEEK2(addr));
        if (channel[n].sample.data != 0) {
            channel[n].duration -= SOUND_TIMER;
        }

        int x = 0;
        if (channel[n].xfade) {
            float delta = 1.0f / (XFADE_SAMPLES);
            while (x < XFADE_SAMPLES * 2) {
                float alpha = x * delta;
                float beta = 1.0f - alpha;
                Sint16 next_a = next_a = next_sample(&channel[n].next_sample);
                Sint16 next_b = 0;
                if (channel[n].sample.data != 0) {
                    next_b = next_sample(&channel[n].sample);
                }
                Sint16 next = alpha * next_a + beta * next_b;
                stream[x++] += next * channel[n].vol_l;
                stream[x++] += next * channel[n].vol_r;
            }
            channel[n].sample = channel[n].next_sample;
            channel[n].xfade = false;
        }
        Sample *sample = &channel[n].sample;
        int direction = 1;
        while (x < len / 2) {
            if (sample->data == 0) {
                break;
            }
            Sint16 next = next_sample(sample);
            stream[x++] += next * channel[n].vol_l;
            stream[x++] += next * channel[n].vol_r;
        }
    }
    for (int i = 0; i < len; i++) {
        stream[i] <<= 4;
    }

    return 1;
}

// #define NOTE_PERIOD (SAMPLE_FREQUENCY * 0x4000 / 11025)
// #define ADSR_STEP (SAMPLE_FREQUENCY / 0xf)

// typedef struct {
// 	Uint8 *addr;
// 	Uint32 count, advance, period, age, a, d, s, r;
// 	Uint16 i, len;
// 	Sint8 volume[2];
// 	Uint8 pitch, repeat;
// } UxnAudio;

// /* clang-format off */

// static Uint32 advances[12] = {
// 	0x80000, 0x879c8, 0x8facd, 0x9837f, 0xa1451, 0xaadc1,
// 	0xb504f, 0xbfc88, 0xcb2ff, 0xd7450, 0xe411f, 0xf1a1c
// };

// static double detunes[256] = {
// 1.0000000000000000, 1.0002256593050698, 1.0004513695322617,
// 1.0006771306930664, 1.0009029427989777, 1.0011288058614922,
// 1.0013547198921082, 1.0015806849023274, 1.0018067009036538,
// 1.002032767907594 , 1.0022588859256572, 1.0024850549693551,
// 1.0027112750502025, 1.0029375461797159, 1.0031638683694153,
// 1.0033902416308227, 1.0036166659754628, 1.0038431414148634,
// 1.004069667960554 , 1.0042962456240678, 1.0045228744169397,
// 1.0047495543507072, 1.004976285436911 , 1.0052030676870944,
// 1.0054299011128027, 1.0056567857255843, 1.0058837215369900,
// 1.006110708558573 , 1.0063377468018897, 1.0065648362784985,
// 1.0067919769999607, 1.0070191689778405, 1.007246412223704 ,
// 1.0074737067491204, 1.0077010525656616, 1.0079284496849015,
// 1.0081558981184175, 1.008383397877789 , 1.008610948974598 ,
// 1.0088385514204294, 1.0090662052268706, 1.0092939104055114,
// 1.0095216669679448, 1.0097494749257656, 1.009977334290572 ,
// 1.0102052450739643, 1.0104332072875455, 1.0106612209429215,
// 1.0108892860517005, 1.0111174026254934, 1.0113455706759138,
// 1.011573790214578 , 1.0118020612531047, 1.0120303838031153,
// 1.0122587578762337, 1.012487183484087 , 1.012715660638304 ,
// 1.0129441893505169, 1.0131727696323602, 1.0134014014954713,
// 1.0136300849514894, 1.0138588200120575, 1.0140876066888203,
// 1.0143164449934257, 1.0145453349375237, 1.0147742765327674,
// 1.0150032697908125, 1.015232314723317 , 1.015461411341942 ,
// 1.0156905596583505, 1.0159197596842091, 1.0161490114311862,
// 1.016378314910953 , 1.0166076701351838, 1.0168370771155553,
// 1.0170665358637463, 1.0172960463914391, 1.017525608710318 ,
// 1.0177552228320703, 1.0179848887683858, 1.0182146065309567,
// 1.0184443761314785, 1.0186741975816487, 1.0189040708931674,
// 1.019133996077738 , 1.0193639731470658, 1.0195940021128593,
// 1.0198240829868295, 1.0200542157806898, 1.0202844005061564,
// 1.0205146371749483, 1.0207449257987866, 1.0209752663893958,
// 1.0212056589585028, 1.0214361035178368, 1.0216666000791297,
// 1.0218971486541166, 1.0221277492545349, 1.0223584018921241,
// 1.0225891065786274, 1.02281986332579  , 1.0230506721453596,
// 1.023281533049087 , 1.0235124460487257, 1.0237434111560313,
// 1.0239744283827625, 1.0242054977406807, 1.0244366192415495,
// 1.0246677928971357, 1.0248990187192082, 1.025130296719539 ,
// 1.0253616269099028, 1.0255930093020766, 1.0258244439078401,
// 1.026055930738976 , 1.0262874698072693, 1.0265190611245079,
// 1.0267507047024822, 1.0269824005529853, 1.027214148687813 ,
// 1.0274459491187637, 1.0276778018576387, 1.0279097069162415,
// 1.0281416643063788, 1.0283736740398595, 1.0286057361284953,
// 1.028837850584101 , 1.0290700174184932, 1.029302236643492 ,
// 1.0295345082709197, 1.0297668323126017, 1.029999208780365 ,
// 1.030231637686041 , 1.030464119041462 , 1.0306966528584645,
// 1.0309292391488862, 1.0311618779245688, 1.0313945691973556,
// 1.0316273129790936, 1.0318601092816313, 1.0320929581168212,
// 1.0323258594965172, 1.0325588134325767, 1.0327918199368598,
// 1.0330248790212284, 1.033257990697548 , 1.0334911549776868,
// 1.033724371873515 , 1.0339576413969056, 1.0341909635597348,
// 1.0344243383738811, 1.034657765851226 , 1.034891246003653 ,
// 1.0351247788430489, 1.0353583643813031, 1.0355920026303078,
// 1.0358256936019572, 1.0360594373081489, 1.0362932337607829,
// 1.0365270829717617, 1.0367609849529913, 1.0369949397163791,
// 1.0372289472738365, 1.0374630076372766, 1.0376971208186156,
// 1.0379312868297725, 1.0381655056826686, 1.0383997773892284,
// 1.0386341019613787, 1.0388684794110492, 1.039102909750172 ,
// 1.0393373929906822, 1.0395719291445176, 1.0398065182236185,
// 1.0400411602399278, 1.0402758552053915, 1.0405106031319582,
// 1.0407454040315787, 1.040980257916207 , 1.0412151647977996,
// 1.041450124688316 , 1.0416851375997183, 1.0419202035439705,
// 1.0421553225330404, 1.042390494578898 , 1.042625719693516 ,
// 1.0428609978888699, 1.043096329176938 , 1.043331713569701 ,
// 1.0435671510791424, 1.0438026417172486, 1.0440381854960086,
// 1.0442737824274138, 1.044509432523459 , 1.044745135796141 ,
// 1.04498089225746  , 1.045216701919418 , 1.0454525647940205,
// 1.0456884808932754, 1.0459244502291931, 1.0461604728137874,
// 1.046396548659074 , 1.046632677777072 , 1.0468688601798024,
// 1.0471050958792898, 1.047341384887561 , 1.0475777272166455,
// 1.047814122878576 , 1.048050571885387 , 1.0482870742491166,
// 1.0485236299818055, 1.0487602390954964, 1.0489969016022356,
// 1.0492336175140715, 1.0494703868430555, 1.0497072096012419,
// 1.0499440858006872, 1.0501810154534512, 1.050417998571596 ,
// 1.0506550351671864, 1.0508921252522903, 1.0511292688389782,
// 1.051366465939323 , 1.0516037165654004, 1.0518410207292894,
// 1.052078378443071 , 1.0523157897188296, 1.0525532545686513,
// 1.0527907730046264, 1.0530283450388465, 1.0532659706834067,
// 1.053503649950405 , 1.053741382851941 , 1.0539791694001188,
// 1.0542170096070436, 1.0544549034848243, 1.0546928510455722,
// 1.0549308523014012, 1.0551689072644284, 1.0554070159467728,
// 1.0556451783605572, 1.0558833945179062, 1.056121664430948 ,
// 1.0563599881118126, 1.0565983655726334, 1.0568367968255465,
// 1.0570752818826903, 1.0573138207562065, 1.057552413458239 ,
// 1.0577910600009348, 1.0580297603964437, 1.058268514656918 ,
// 1.0585073227945128, 1.0587461848213857, 1.058985100749698 ,
// 1.0592240705916123,
// };

// static UxnAudio uxn_audio[POLYPHONY];

// /* clang-format on */

// static Sint32
// envelope(UxnAudio *c, Uint32 age)
// {
// 	if(!c->r) return 0x0888;
// 	if(age < c->a) return 0x0888 * age / c->a;
// 	if(age < c->d) return 0x0444 * (2 * c->d - c->a - age) / (c->d - c->a);
// 	if(age < c->s) return 0x0444;
// 	if(age < c->r) return 0x0444 * (c->r - age) / (c->r - c->s);
// 	c->advance = 0;
// 	return 0x0000;
// }

// int
// audio_render(int instance, Sint16 *sample, Sint16 *end)
// {
//     /*
// 	UxnAudio *c = &uxn_audio[instance];
// 	Sint32 s;
// 	if(!c->advance || !c->period) return 0;
// 	while(sample < end) {
// 		c->count += c->advance;
// 		c->i += c->count / c->period;
// 		c->count %= c->period;
// 		if(c->i >= c->len) {
// 			if(!c->repeat) {
// 				c->advance = 0;
// 				break;
// 			}
// 			c->i %= c->len;
// 		}
// 		s = (Sint8)(c->addr[c->i] + 0x80) * envelope(c, c->age++);
// 		*sample++ += s * c->volume[0] / 0x180;
// 		*sample++ += s * c->volume[1] / 0x180;
// 	}
// 	if(!c->advance) audio_finished_handler(instance);
// 	return 1;
// 	*/
// }

void
audio_start(int idx, Uint8 *d, Uxn *u)
{
    Uint16 duration = PEEK2(d + 0x2);
    Uint8 off = d[0xf] == 0xff;

    if (!off) {
        // Sample.
        Uint16 addr = PEEK2(d + 0xa);
        Uint8 *data = &u->ram[addr];
        Uint16 len = PEEK2(d + 0x8);
        Uint8 volume = d[0xd];
        bool loop = !!(d[0xf] & 0x80);

        // Pitch.
        Uint8 pitch = d[0xf] & 0x7f;

        // ADSR.
        Uint8 attack = d[0x4];
        Uint8 decay = d[0x5];
        Uint8 sustain = d[0x6];
        Uint8 release = d[0x7];

        note_on(&channel[idx], duration, data, len, volume, attack, decay, sustain, release, pitch, loop);
    } else {
        note_off(&channel[idx], duration);
    }
}
