#include "../uxn.h"
#include "audio.h"

/*
Copyright (c) 2021-2023 Devine Lu Linvega, Andrew Alderwick

Permission to use, copy, modify, and distribute this software for any
purpose with or without fee is hereby granted, provided that the above
copyright notice and this permission notice appear in all copies.

THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
WITH REGARD TO THIS SOFTWARE.
*/

#define NOTE_PERIOD (SAMPLE_FREQUENCY * 0x4000 / 11025)
#define ADSR_STEP (SAMPLE_FREQUENCY / 0xf)

typedef struct {
	Uint8 *addr;
	Uint32 count, advance, period, age, a, d, s, r;
	Uint16 i, len;
	Sint8 volume[2];
	Uint8 pitch, repeat;
} UxnAudio;

/* clang-format off */

static Uint32 advances[12] = {
	0x80000, 0x879c8, 0x8facd, 0x9837f, 0xa1451, 0xaadc1,
	0xb504f, 0xbfc88, 0xcb2ff, 0xd7450, 0xe411f, 0xf1a1c
};

static double detunes[256] = {
1.0000000000000000, 1.0002256593050698, 1.0004513695322617, 
1.0006771306930664, 1.0009029427989777, 1.0011288058614922, 
1.0013547198921082, 1.0015806849023274, 1.0018067009036538, 
1.002032767907594 , 1.0022588859256572, 1.0024850549693551, 
1.0027112750502025, 1.0029375461797159, 1.0031638683694153, 
1.0033902416308227, 1.0036166659754628, 1.0038431414148634, 
1.004069667960554 , 1.0042962456240678, 1.0045228744169397, 
1.0047495543507072, 1.004976285436911 , 1.0052030676870944, 
1.0054299011128027, 1.0056567857255843, 1.0058837215369900, 
1.006110708558573 , 1.0063377468018897, 1.0065648362784985, 
1.0067919769999607, 1.0070191689778405, 1.007246412223704 , 
1.0074737067491204, 1.0077010525656616, 1.0079284496849015, 
1.0081558981184175, 1.008383397877789 , 1.008610948974598 , 
1.0088385514204294, 1.0090662052268706, 1.0092939104055114, 
1.0095216669679448, 1.0097494749257656, 1.009977334290572 , 
1.0102052450739643, 1.0104332072875455, 1.0106612209429215, 
1.0108892860517005, 1.0111174026254934, 1.0113455706759138, 
1.011573790214578 , 1.0118020612531047, 1.0120303838031153, 
1.0122587578762337, 1.012487183484087 , 1.012715660638304 , 
1.0129441893505169, 1.0131727696323602, 1.0134014014954713, 
1.0136300849514894, 1.0138588200120575, 1.0140876066888203, 
1.0143164449934257, 1.0145453349375237, 1.0147742765327674, 
1.0150032697908125, 1.015232314723317 , 1.015461411341942 , 
1.0156905596583505, 1.0159197596842091, 1.0161490114311862, 
1.016378314910953 , 1.0166076701351838, 1.0168370771155553, 
1.0170665358637463, 1.0172960463914391, 1.017525608710318 , 
1.0177552228320703, 1.0179848887683858, 1.0182146065309567, 
1.0184443761314785, 1.0186741975816487, 1.0189040708931674, 
1.019133996077738 , 1.0193639731470658, 1.0195940021128593, 
1.0198240829868295, 1.0200542157806898, 1.0202844005061564, 
1.0205146371749483, 1.0207449257987866, 1.0209752663893958, 
1.0212056589585028, 1.0214361035178368, 1.0216666000791297, 
1.0218971486541166, 1.0221277492545349, 1.0223584018921241, 
1.0225891065786274, 1.02281986332579  , 1.0230506721453596, 
1.023281533049087 , 1.0235124460487257, 1.0237434111560313, 
1.0239744283827625, 1.0242054977406807, 1.0244366192415495, 
1.0246677928971357, 1.0248990187192082, 1.025130296719539 , 
1.0253616269099028, 1.0255930093020766, 1.0258244439078401, 
1.026055930738976 , 1.0262874698072693, 1.0265190611245079, 
1.0267507047024822, 1.0269824005529853, 1.027214148687813 , 
1.0274459491187637, 1.0276778018576387, 1.0279097069162415, 
1.0281416643063788, 1.0283736740398595, 1.0286057361284953, 
1.028837850584101 , 1.0290700174184932, 1.029302236643492 , 
1.0295345082709197, 1.0297668323126017, 1.029999208780365 , 
1.030231637686041 , 1.030464119041462 , 1.0306966528584645, 
1.0309292391488862, 1.0311618779245688, 1.0313945691973556, 
1.0316273129790936, 1.0318601092816313, 1.0320929581168212, 
1.0323258594965172, 1.0325588134325767, 1.0327918199368598, 
1.0330248790212284, 1.033257990697548 , 1.0334911549776868, 
1.033724371873515 , 1.0339576413969056, 1.0341909635597348, 
1.0344243383738811, 1.034657765851226 , 1.034891246003653 , 
1.0351247788430489, 1.0353583643813031, 1.0355920026303078, 
1.0358256936019572, 1.0360594373081489, 1.0362932337607829, 
1.0365270829717617, 1.0367609849529913, 1.0369949397163791, 
1.0372289472738365, 1.0374630076372766, 1.0376971208186156, 
1.0379312868297725, 1.0381655056826686, 1.0383997773892284, 
1.0386341019613787, 1.0388684794110492, 1.039102909750172 , 
1.0393373929906822, 1.0395719291445176, 1.0398065182236185, 
1.0400411602399278, 1.0402758552053915, 1.0405106031319582, 
1.0407454040315787, 1.040980257916207 , 1.0412151647977996, 
1.041450124688316 , 1.0416851375997183, 1.0419202035439705, 
1.0421553225330404, 1.042390494578898 , 1.042625719693516 , 
1.0428609978888699, 1.043096329176938 , 1.043331713569701 , 
1.0435671510791424, 1.0438026417172486, 1.0440381854960086, 
1.0442737824274138, 1.044509432523459 , 1.044745135796141 , 
1.04498089225746  , 1.045216701919418 , 1.0454525647940205, 
1.0456884808932754, 1.0459244502291931, 1.0461604728137874, 
1.046396548659074 , 1.046632677777072 , 1.0468688601798024, 
1.0471050958792898, 1.047341384887561 , 1.0475777272166455, 
1.047814122878576 , 1.048050571885387 , 1.0482870742491166, 
1.0485236299818055, 1.0487602390954964, 1.0489969016022356, 
1.0492336175140715, 1.0494703868430555, 1.0497072096012419, 
1.0499440858006872, 1.0501810154534512, 1.050417998571596 , 
1.0506550351671864, 1.0508921252522903, 1.0511292688389782, 
1.051366465939323 , 1.0516037165654004, 1.0518410207292894, 
1.052078378443071 , 1.0523157897188296, 1.0525532545686513, 
1.0527907730046264, 1.0530283450388465, 1.0532659706834067, 
1.053503649950405 , 1.053741382851941 , 1.0539791694001188, 
1.0542170096070436, 1.0544549034848243, 1.0546928510455722, 
1.0549308523014012, 1.0551689072644284, 1.0554070159467728, 
1.0556451783605572, 1.0558833945179062, 1.056121664430948 , 
1.0563599881118126, 1.0565983655726334, 1.0568367968255465, 
1.0570752818826903, 1.0573138207562065, 1.057552413458239 , 
1.0577910600009348, 1.0580297603964437, 1.058268514656918 , 
1.0585073227945128, 1.0587461848213857, 1.058985100749698 , 
1.0592240705916123, 
};

static UxnAudio uxn_audio[POLYPHONY];

/* clang-format on */

static Sint32
envelope(UxnAudio *c, Uint32 age)
{
	if(!c->r) return 0x0888;
	if(age < c->a) return 0x0888 * age / c->a;
	if(age < c->d) return 0x0444 * (2 * c->d - c->a - age) / (c->d - c->a);
	if(age < c->s) return 0x0444;
	if(age < c->r) return 0x0444 * (c->r - age) / (c->r - c->s);
	c->advance = 0;
	return 0x0000;
}

int
audio_render(int instance, Sint16 *sample, Sint16 *end)
{
	UxnAudio *c = &uxn_audio[instance];
	Sint32 s;
	if(!c->advance || !c->period) return 0;
	while(sample < end) {
		c->count += c->advance;
		c->i += c->count / c->period;
		c->count %= c->period;
		if(c->i >= c->len) {
			if(!c->repeat) {
				c->advance = 0;
				break;
			}
			c->i %= c->len;
		}
		s = (Sint8)(c->addr[c->i] + 0x80) * envelope(c, c->age++);
		*sample++ += s * c->volume[0] / 0x180;
		*sample++ += s * c->volume[1] / 0x180;
	}
	if(!c->advance) audio_finished_handler(instance);
	return 1;
}

void
audio_start(int instance, Uint8 *d, Uxn *u)
{
	UxnAudio *c = &uxn_audio[instance];
	Uint8 pitch = d[0xf] & 0x7f;
	Uint8 detune = d[0x5];
	Uint16 addr = PEEK2(d + 0xc), adsr = PEEK2(d + 0x8);
	c->len = PEEK2(d + 0xa);
	if(c->len > 0x10000 - addr)
		c->len = 0x10000 - addr;
	c->addr = &u->ram[addr];
	c->volume[0] = d[0xe] >> 4;
	c->volume[1] = d[0xe] & 0xf;
	c->repeat = !(d[0xf] & 0x80);
	if(pitch < 108 && c->len)
		c->advance = (Uint32)((double)(advances[pitch % 12]) * detunes[detune]) >> (8 - pitch / 12);
	else {
		c->advance = 0;
		return;
	}
	c->a = ADSR_STEP * (adsr >> 12);
	c->d = ADSR_STEP * (adsr >> 8 & 0xf) + c->a;
	c->s = ADSR_STEP * (adsr >> 4 & 0xf) + c->d;
	c->r = ADSR_STEP * (adsr >> 0 & 0xf) + c->s;
	c->age = 0;
	c->i = 0;
	if(c->len <= 0x100) /* single cycle mode */
		c->period = NOTE_PERIOD * 337 / 2 / c->len;
	else /* sample repeat mode */
		c->period = NOTE_PERIOD;
}

Uint8
audio_get_vu(int instance)
{
	UxnAudio *c = &uxn_audio[instance];
	int i;
	Sint32 sum[2] = {0, 0};
	if(!c->advance || !c->period) return 0;
	for(i = 0; i < 2; i++) {
		if(!c->volume[i]) continue;
		sum[i] = 1 + envelope(c, c->age) * c->volume[i] / 0x800;
		if(sum[i] > 0xf) sum[i] = 0xf;
	}
	return (sum[0] << 4) | sum[1];
}

Uint16
audio_get_position(int instance)
{
	UxnAudio *c = &uxn_audio[instance];
	return c->i;
}
